/*======================================================================
*** printing.c ***
* (c) Alma L. Juarez-Dominguez, Richard Fanson
* C functions for the SMV output creation.
*
* Created: August 2007
* Last modified: January 6, 2008
*======================================================================*/

#include "data.h"
#include "structs.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

extern FILE* tree;
extern FILE* lists;
extern FILE* DoneReady;

extern list* stateList;
extern list* transList;
extern list* juncList;
extern list* eventList;
extern list* dataList;
extern list* stateTree;

extern int tabs;
extern int tabsMAX;
extern int ANDstates;
extern int ANDlevel;
extern int Readyprinted;
extern int ReadyOR; 

int eventsCnt = 0;

/*****************************************************************************************
*******************     FUNCTIONS FOR PRINTING IN SMV     ********************************
*****************************************************************************************/

/**************************************************************
Print all the data variable declaration information
**************************************************************/

void printInputOutput()
{
	node* temp;
	char buffer[100]; // buffer used to eat bad input since scanf leaves it on the stream
	/*char storedVars[1000] = "\n";*/
	int lowVal = 0;
	int highVal = 0;
	int readSuccess = 0;
	int prompt = 0;

	printEvents();
	
	// get ranges for all the input variables
	temp = dataList->head;
	while(temp != NULL)
	{
		// note that strcmp returns 0 if the strings match, hence the negation
		if(!strcmp(temp->scope, "INPUT_DATA"))
		{	
			if(!strcmp(temp->triggerData, "boolean"))
				printf("\tINPUT %s : %s;\n", temp->labelString, temp->triggerData);

			else 
			{
				if(!strcmp(temp->triggerData, "inherited")) 
				{
					if(!prompt)
					{
						fprintf(stderr, "\nInherited type data found.\nPlease enter possible range of values\n");
						prompt = 1;
					}
					
					fprintf(stderr, "\nInput \"%s\" of type \"%s\" found.\n", temp->labelString, temp->triggerData);
	
					readSuccess = 0;
					fprintf(stderr, "Enter lowest possible value: ");
					readSuccess = scanf("%d", &lowVal);
					
					// if the types do not match in scanf, it will return 0
					// so do this while loop until the types match properly
					while(readSuccess == 0)
					{
						fprintf(stderr, "Bad input. Try again: ");
						gets(buffer); // eat the old string
						readSuccess = scanf("%d", &lowVal);
					}
	
					readSuccess = 0;
					fprintf(stderr, "Enter highest possible value: ");
					readSuccess = scanf("%d", &highVal);
					while(readSuccess == 0)
					{
						fprintf(stderr, "Bad input. Try again: ");
						gets(buffer); // eat the old string
						readSuccess = scanf("%d", &highVal);
					}
	
					printf("\tINPUT %s : %d..%d;\n", temp->labelString, lowVal, highVal);
				}	
				else
				{
					printf("\tINPUT %s : %d..%d;\n", temp->labelString, temp->min_range, temp->max_range);
				}	
			}
		}
		temp = temp->next;
	}

	// get ranges for all output data
	temp = dataList->head;
	while(temp != NULL)
	{
		if(!strcmp(temp->scope, "OUTPUT_DATA"))
		{
			if(!strcmp(temp->triggerData, "boolean"))
				printf("\tOUTPUT %s : %s;\n", temp->labelString, temp->triggerData);

			else 
			{
				if(!strcmp(temp->triggerData, "inherited")) 
				{
					if(!prompt)
					{
						fprintf(stderr, "\nNon-boolean data found.\nPlease enter possible range of values\n");
						prompt = 1;
					}
					
					fprintf(stderr, "\nOutput \"%s\" of type \"%s\" found.\n", temp->labelString, temp->triggerData);
	
					
					readSuccess = 0;
					fprintf(stderr, "Enter lowest possible value: ");
					readSuccess = scanf("%d", &lowVal);
					while(readSuccess == 0)
					{
						fprintf(stderr, "Bad input. Try again: ");
						gets(buffer); // eat the old string
						readSuccess = scanf("%d", &lowVal);
					}
					
					readSuccess = 0;
					fprintf(stderr, "Enter highest possible value: ");
					readSuccess = scanf("%d", &highVal);
					while(readSuccess == 0)
					{
						fprintf(stderr, "Bad input. Try again: ");
						gets(buffer); // eat the old string
						readSuccess = scanf("%d", &highVal);
					}
	
					printf("\tOUTPUT %s : %d..%d;\n", temp->labelString, lowVal, highVal);
				}	
				else
				{
					printf("\tOUTPUT %s : %d..%d;\n", temp->labelString, temp->min_range, temp->max_range);
				}		
			}
			// For outputs that are parameterized command (naming convention: name starting with "set_"),
			// translator declares a boolean variable that represents the presence or absence of the command,
			// with naming convention NAME_req
			// NOTE (Aug 12, 2010): Also, for outputs that are parameterized commands, adding a boolean 
			// variable with nameing convention NAME_hreq to hold information of the req when a feature 
			// contains AND-states, so the info about requests is available when it's time to check for   
			// the occurence of the command (at stable)
			if(!strncmp(temp->labelString, "set_",4))
			{
				char buffer[500];
				int i = 4; // for the labelString
				int j = 0; // for the buffer

				while(temp->labelString[i] != '\0')
				{
					buffer[j] = temp->labelString[i];
					i++;
					j++;
				}	
				buffer[j++] = '_';
				buffer[j++] = 'r';
				buffer[j++] = 'e';
				buffer[j++] = 'q';
				buffer[j] = '\0';
				node * new = createNode();	
				addToList(dataList, new);
				// name
				addLabelString(dataList, buffer);
				// Scope
				addScope(dataList, "LOCAL_DATA");
				// Datatype
				buffer[0] = '"';
				buffer[1] = 'b';
				buffer[2] = 'o';
				buffer[3] = 'o';
				buffer[4] = 'l';
				buffer[5] = 'e';
				buffer[6] = 'a';
				buffer[7] = 'n';
				buffer[8] = '"';
				buffer[9] = '\0';
				addTriggerData(dataList, buffer);

				buffer[0] = '0';
				buffer[1] = '\0';
				addInitVal(dataList, buffer);
				
			}
		}
		temp = temp->next;
	}


	// get ranges for all local data
	temp = dataList->head;
	while(temp != NULL)
	{
		if(!strcmp(temp->scope, "LOCAL_DATA"))
		{
			if(!strcmp(temp->triggerData, "boolean"))
				printf("\t%s : %s;\n", temp->labelString, temp->triggerData);

			else 
			{
				if(!strcmp(temp->triggerData, "inherited")) 
				{
					if(!prompt)
					{
						fprintf(stderr, "\nNon-boolean data found.\nPlease enter possible range of values\n");
						prompt = 1;
					}
					
					fprintf(stderr, "\nLocal data \"%s\" of type \"%s\" found.", temp->labelString, temp->triggerData);
	
					readSuccess = 0;
					fprintf(stderr, "Enter lowest possible value: ");
					readSuccess = scanf("%d", &lowVal);
					while(readSuccess == 0)
					{
						fprintf(stderr, "Bad input. Try again: ");
						gets(buffer); // eat the old string
						readSuccess = scanf("%d", &lowVal);
					}
					
					readSuccess = 0;
					fprintf(stderr, "Enter highest possible value: ");
					readSuccess = scanf("%d", &highVal);
					while(readSuccess == 0)
					{
						fprintf(stderr, "Bad input. Try again: ");
						gets(buffer); // eat the old string
						readSuccess = scanf("%d", &highVal);
					}
					
					printf("\t%s : %d..%d;\n", temp->labelString, lowVal, highVal);
				}	
				else
				{
					printf("\t%s : %d..%d;\n", temp->labelString, temp->min_range, temp->max_range);
				}	

			}
		}
		temp = temp->next;
	}
	
}



/**************************************************************
Print all the INPUT events in an enumerated form
**************************************************************/

void printEvents()
{
	eventsCnt = 0;
	int n =0;

	// check to see if we have any INPUT events
	node* temp = eventList->head;
	while (temp != NULL)
	{
		// if strstr finds "INPUT" it returns a pointer to it
		if(strstr(temp->scope, "INPUT") != NULL)
			eventsCnt++;
		temp = temp->next;
	}

	// if we found INPUT events declare them
	if (eventsCnt)
		printf("\tINPUT %sevents : {", stateList->head->labelString);  
	
	// loop through again to declare all the input events
	temp = eventList->head;
	while (temp != NULL)
	{
		if(strstr(temp->scope, "INPUT") != NULL)
			if (n==0)
			{	printf("%s", temp->labelString); n++; }
			else
			{	printf(",%s", temp->labelString); n++; }
		else
			fprintf(stderr, "\n\nOUTPUT or LOCAL events found! These are not accounted for yet.\n\n");
		temp = temp->next;
	}
	
	if (eventsCnt)
		printf("};\n");

}


/**************************************************************
 Print the trans hierarchy in enumerated forms if required 
 **************************************************************/

void printTrans(int trans_hierarchy)
{
	node* temptrans;	
	node* parent = stateList->head; // start at the head which should be the main chart (id: 2)
	int printing=1;
	
	if (trans_hierarchy)
	{
		while(parent != NULL)
		{
			if (parent->childId != 0) 
			{
				//if ((parent->parentId == 0) || !strcmp(parent->type, "OR_STATE"))
				//if (!strcmp(parent->child->type, "OR_STATE"))	
				//{	
				if (((parent->parentId == 0) && (!strcmp(parent->child->type, "OR_STATE"))) || ((parent->parentId != 0) && !strcmp(parent->type, "AND_STATE")))
				{	
					printf("\tTrans%s : {t0, tn", parent->labelString);
					printing=1;
				
					node* child = parent;
					while ((child != NULL) && printing)
					{	
						if (child->child == NULL)
							if (child->rightSib != NULL)
								child = child ->rightSib;
							else
								child = child->parent->rightSib;
						else
							if (!strcmp(child->child->type, "AND_STATE"))
								child = child->rightSib;
							else
							// if not AND_STATE, it must be OR_STATE!
							//if (!strcmp(child->child->type, "OR_STATE"))
							{	
								temptrans = transList->head;
								while (temptrans != NULL)
								{
									// trans-parent is null when trans is a default transition, we are not interested in deafult trans!
									if (temptrans->parent != NULL)
										if (temptrans->parent->parent->id == child->id)
											if (temptrans->parent->parent->id == temptrans->child->parent->id)
												printf(",t%d", temptrans->id);
											else
											{
												node* temptrans2 = temptrans->parent;	
												while (temptrans2 != NULL)	
												{	
													if (temptrans2->parent == NULL)
														printf(",t%d", temptrans->id);
													else
														if (temptrans2->parent->id == temptrans->child->parent->id)
															break;
													temptrans2 = temptrans2->parent;
												}	
											}
										else
											if (temptrans->child->parent->id == child->id)
												if (temptrans->parent->parent->id == temptrans->child->parent->id)
													printf(",t%d", temptrans->id);
												else
												{
													node* temptrans2 = temptrans->child;	
													while (temptrans2 != NULL)	
													{	
														if (temptrans2->parent == NULL)
															printf(",t%d", temptrans->id);
														else
															if (temptrans2->parent->id == temptrans->parent->parent->id)
																break;
														temptrans2 = temptrans2->parent;
													}	
												}
									temptrans = temptrans->next;
								}	
								child = child->child;
							}
							if ((child==NULL) || ((child->parentId != 0) && !strcmp(child->type, "AND_STATE"))) printing=0;
							else printing=1;
					}	
				//}
					printf("};\n");
				}
			}
		
			parent = parent->next;
		}
	
		printf("\n");
	}
	// no hierarchy required
	else
	{	
		// Jan 6, 2009
		// Enumerated type variable to define transitions taken, which has the form TransFNAME : {tn,t#}
		// where # in t# comes from transition id's (from .mdl file).
		// tn is used to denote do-nothing transitions, or initialization of transitions (has done
		// nothing) -- Before, used go use TDN#, where # came from states id's, but tn simplifies 
		// properties written, no need to differentiate which do-nothing transition was taken, only
		// need to know that a do-nothing transition was taken.
		
		temptrans = transList->head;
		printf("\tTrans%s : {tn", stateList->head->labelString);
		while(temptrans != NULL)
		{
			// trans-parent is null when trans is a default transition, we are not interested in deafult trans!
			if (temptrans->parent != NULL)
				printf(",t%d", temptrans->id); 
			temptrans = temptrans->next;
		}	
		printf("};");
		
		printf("\n");
	}	
	
}



/**************************************************************
Print the state hierarchy in enumerated forms
**************************************************************/

void printStates()
{
	node* parent = stateList->head; // start at the head which should be the main chart (id: 2)
	node* child = NULL;

	while(parent != NULL)
	{
		if(parent->childId != 0)
		{
			printf("\tstate%s : {", parent->labelString);
			child = parent->child;
			
			while(child != NULL)
			{
				// AND-states always have children, so they cannot be basic states
				if (!strcmp(child->type, "AND_STATE"))
				{	
					// if its the first child print only the child's name
					if(child->id == parent->childId)
						printf("state%s%s", child->labelString,stateList->head->labelString);
					
					// if its another child print out a comma and then the child's name
					else
						printf(",state%s%s", child->labelString,stateList->head->labelString);
				}
				// Basic state if an OR-state is childless
				// Jul 25, 2011: No distinction between basic and superstates
				else
				{
					// if its the first child print only the child's name
					if(child->id == parent->childId)
						printf("state%s%s", child->labelString,stateList->head->labelString);
						
					// if its another child print out a comma and then the child's name
					else
						printf(",state%s%s", child->labelString,stateList->head->labelString);
				}	

				child = child->rightSib;
			}

			printf("};\n");
		}

		parent = parent->next;
	}

	printf("\n");
}


/**************************************************************
Initialize the hierarchy of the states
**************************************************************/

void printInits(int trans_hierarchy)
{
	node* state = NULL;
	node* trans = NULL;
	node* child = NULL;
	node* defaultChild = NULL;

	// initialize states
	state = stateTree->head;
	while(state != NULL)
	{
		// if the state is a parent
		if(state->childId != 0)
		{
				// find the child who has a transition with no src (is the default transition)
				child = state->child;
				while (child != NULL)	
				{
					trans = transList->head;
					while (trans != NULL)
					{
						if (trans->dst == child->id && trans->src == 0)
							defaultChild = child;
						trans = trans->next;
					}
					child = child->rightSib;
				}
				
				// default transition for OR_STATE
				if (defaultChild != NULL)
				{	
					printf("\tinit(state%s) := state%s%s;\n", state->labelString, defaultChild->labelString,stateList->head->labelString);
				}
				else
				{
					child = state->child;
					while (child != NULL)	
					{
						if (!strcmp(child->type, "AND_STATE") && child->execOrder == 1)
							defaultChild = child;
						child = child->rightSib;
					}
					// default transition for AND_STATE (state with execution order 1)
					if (defaultChild != NULL)
						printf("\tinit(state%s) := state%s%s;\n", state->labelString, defaultChild->labelString,stateList->head->labelString);
				}
					
				defaultChild = NULL;
		}
		state = state->next;
	}
	
	printf("\n");
	
	// initialize local data if it exists
	node* temp = dataList->head;
	while(temp != NULL)
	{
		if(!strcmp(temp->scope, "LOCAL_DATA"))
		{
			// For _req locals, value associated at creation is 0 (which means false in SMV)	
			// For other locals, it associates the value defined in Stateflow, OR 0 otherwise (default value)
			printf("\tinit(%s) := %d;\n", temp->labelString, temp->init_val);
		}
		temp = temp->next;
	}	
	
	state = stateList->head;
	// Initialize hierarchy of transitions if it exist
	if (trans_hierarchy)
	{
		while(state != NULL)
		{
			if (state->childId != 0) 
				if (((state->parentId == 0) && (!strcmp(state->child->type, "OR_STATE"))) || ((state->parentId != 0) && !strcmp(state->type, "AND_STATE")))	
				{	
					printf("\tinit(Trans%s) := tn;\n", state->labelString);
				}	
			
			state = state->next;
		}
		
		printf("\n");
	}
	// else-> no hierarchy required, then no initialization of trans is needed. 
	// However, we need initialization of trans_on variable to 0
	else
		printf("\tinit(Trans%s) := tn;\n", state->labelString);
		
	// print initialization for transition taken variable
	//printf("\tinit(Trans%s) := tn;\n", stateList->head->labelString); 
	//printf("\n");
}


/**************************************************************
 Print condition stable for the current feature hierarchically
 **************************************************************/

// parent = stateList->head the first time the function is called
// parent = all children of parent if children are OR-states
// parent = parent->child if children are AND-states (find the ones with execution order 1 hierarchically)
void printStable(node * parent)
{
	char buffer_stable[500];
	
	if (ANDstates==0) 
	{				
		// For the case when there's no AND-states, this is the only definition of stable
		strcpy(buffer_stable,"DEFINE stable");
		strcat(buffer_stable," := 1");	
	}	
	// There are AND-states, check if the current children are AND or OR states
	else
	{
		// If it's the first call to the function, define main stable
		if (parent->parentId == 0)
		{	
			strcpy(buffer_stable,"DEFINE stable");
			strcat(buffer_stable," := ");	
		} 
		// If current parent is childless, it's an OR state, print condition equal to being at the state (eg, state = me)
		// Otherwise, for the children, check depending on type (OR or AND states) 
		// For successive calls, name stable as stableSTATENAME
		else
		{
			strcpy(buffer_stable,"DEFINE stable");
			strcat(buffer_stable,parent->labelString);
			strcat(buffer_stable," := ");
		}
		
		// Do something only if a parent has children
		if (parent->child!=NULL) 
		{
				
			node * temp = parent->child;
			// If state's child is an AND-state
			if (!strcmp(temp->type, "AND_STATE"))
			{	
				int condnum = 0;
				// Create stable condition WRT states -> Need *and* in conditions, not or's!!!!
				// First, update outwards, so the definition of stable is hierarchical
				// Second, for any AND-state: (1) Find sibling with execution order 1, 2) conjunction 
				// with all of its children with execution order 1
				
				
				// Updating inwards
				temp = parent->child;
				while (temp != NULL) 
				{
					
					if (!strcmp(temp->type, "AND_STATE"))
					{	
						// Find AND-state sibling with execution order 1
						node* workingstate = temp; 
						
						// Check for all the rightSib as long as it's not NULL until we find AND-state sibling with execution order 1
						while (workingstate != NULL)
						{
							if (workingstate->execOrder == 1)
								temp = workingstate;
							workingstate = workingstate->rightSib;
						}	
						// Current value of temp is AND-state with execution order 1, so include temp in condition.
						if (condnum>0)
						{	strcat(buffer_stable," & "); }
					
						strcat(buffer_stable,"(state");
						strcat(buffer_stable,temp->parent->labelString);
						strcat(buffer_stable," = state");
						strcat(buffer_stable,temp->labelString);
						strcat(buffer_stable,stateList->head->labelString);
					
						strcat(buffer_stable,")");
						condnum++;
					}	
				
						/*if (temp->child!=NULL)
							if (!strcmp(temp->child->type, "AND_STATE"))
								temp = temp->child; 
							else
								if (temp->child->child==NULL) 
									temp = NULL;
								else
								{	printStable(temp->child); temp = NULL; }
						else
							temp = temp->child; // NULL!!!*/
						temp = temp->child;
									
				}
				
				// Updating outwards
				temp = parent;
				while ((temp != NULL) && (temp->parentId != 0))
				{
					if (condnum>0)
					{	strcat(buffer_stable," & "); }
					
					strcat(buffer_stable,"(state");
					strcat(buffer_stable,temp->parent->labelString);
					strcat(buffer_stable," = state");					
					strcat(buffer_stable,temp->labelString);
					strcat(buffer_stable,stateList->head->labelString);
					
					strcat(buffer_stable,")");
					condnum++;
					
					temp = temp->parent;
				}	
					
			}
			// If state's child is an OR-state
			// Print disjunction of all the transtitions within the parent, disjuncted with the stable condition of each sibling
			else
			{
				// A parent with childless children must be an OR-state. 
				// If it's an OR-state, print condition equal to myself (name of macro is stableSTATENAME)
				// AND update upward hierarchically! Note, we need conjunctions because we need the whole hierarchy to hold
				if (temp->child==NULL) 
				{
					
					// Updating outwards from current temp->parent
					node* tt = temp->parent;
					int condnum = 0;
					while ((tt != NULL) && (tt->parent !=NULL))// (tt->parentId != 0))
					{
						if (condnum>0)
						{	strcat(buffer_stable," & "); }
						
						strcat(buffer_stable,"(state");
						strcat(buffer_stable,tt->parent->labelString);
						if ((!strcmp(temp->type, "AND_STATE")) || ((!strcmp(temp->type, "OR_STATE")) && (temp->child != NULL)))
						strcat(buffer_stable," = state"); 
						strcat(buffer_stable,tt->labelString);
						strcat(buffer_stable,stateList->head->labelString);
						
						strcat(buffer_stable,")");
						condnum++;
						
						tt = tt->parent;
					}
				}
				// If parent has children, create stable macros appropriately 
				else
				{	
					// (1) Find all trans whose src is any of the children of the current state if src and dst in the same hierarchy level
					// If incoming or outgoing transitions at different hierarchy levels:
					// If incoming, use the parent (src); If outgoing, use child (dst)
					node* workingstate = temp;
					int condnum = 0;
								
					while (workingstate != NULL) 
					{
						node* trans = transList->head;
						while(trans != NULL)
						{	
							if (trans->parent != NULL)
								// Check if src and dst are int he same hierarchy level
								if (trans->parent->parent->id == trans->child->parent->id)
								{	
									//trans->src same as the id of the child, check all siblings following the rightSib link 
									if (trans->src == workingstate->id)
									{
										// NOTE: Already printed the conjunction, for all other states, print disjunction
										if (condnum>0)
										{	strcat(buffer_stable," | "); }
										
										char tid[5]; 
										
										strcat(buffer_stable,"(Trans");
										strcat(buffer_stable,stateList->head->labelString);
										strcat(buffer_stable," = t");
										sprintf(tid, "%d", trans->id);
										strcat(buffer_stable,tid);
										
										strcat(buffer_stable," )");
																		
										condnum++;
									}	
								}	
								else
								{
									// Find outermost end of trans: Start with trans->parent. Update with trans->parent.
									// If reach NULL while checking the trans' parent, trans->parent (src) is outermost, else, trans->child (dst) is outermost
									// If outermost is trans->parent, make outerId= trans->src; If outermost is trans->child, make outerId= trans->dst
									node* outermost = trans->parent;	
									int outerId;
									while (outermost != NULL)	
									{	
										if (outermost->parent == NULL)
										{	outerId = trans->src; break; }
										else
											if (outermost->parent->id == trans->child->parent->id)
											{	outerId = trans->dst; break; }
										outermost = outermost->parent;
									}
																		
									//outermost is src and has the same as the id of the child, print info, but check checking wrt trans for rightSib
									if (outerId == workingstate->id)
									{
										if (condnum>0)
										{	strcat(buffer_stable," | "); }
										
										char tid[5]; 
										
										strcat(buffer_stable,"(Trans");
										strcat(buffer_stable,stateList->head->labelString);
										strcat(buffer_stable," = t");
										sprintf(tid, "%d", trans->id);
										strcat(buffer_stable,tid);
										
										strcat(buffer_stable," )");
																				
										condnum++;
									}
								}	
						
							trans = trans->next;	
						}	
						
						workingstate = workingstate->rightSib;
					}	
			
					// (2) Find all children and add a stable condition for each of them, calling function recursivelly for all of them
					// NOTE: Not call on children if they are childless!!! child->child==NULL
					while (temp != NULL) 
					{
						// Current value of temp is OR-state child of parent, so include temp in condition.
						if (condnum>0)
						{	strcat(buffer_stable," | "); }
				
						strcat(buffer_stable,"(stable");
						strcat(buffer_stable,temp->labelString);
						strcat(buffer_stable,")");
				
						condnum++;
				
						/*if (temp->child!=NULL)
							if (temp->child->child==NULL)
								temp = temp->rightSib; 
							else
								{	printStable(temp->child); temp = temp->rightSib; }
						else
						{	printStable(temp->child); temp = NULL; }*/
						// temp is already parent->child... if we use temp->child we are skipping one hierarchy level!
						printStable(temp); temp = temp->rightSib;
				
					}
					//strcat(buffer_stable," )");		
				}
			}	
		}
		// update parent with no children upwards
		else
		{
			// Updating outwards
			node* temp = parent;
			int condnum=0;
			
			while ((temp != NULL) && (temp->parentId != 0))
			{
				if (condnum>0)
				{	strcat(buffer_stable," & "); }
				
				strcat(buffer_stable,"(state");
				strcat(buffer_stable,temp->parent->labelString);
				strcat(buffer_stable," = state"); 
				strcat(buffer_stable,temp->labelString);
				strcat(buffer_stable,stateList->head->labelString);
				
				strcat(buffer_stable,")");
				condnum++;
				
				temp = temp->parent;
			}	
		}	
	}
	
	strcat(buffer_stable,";");
	
	buffer_stable[strlen(buffer_stable)]='\0';
		
	printf("\n");
	align(); printf("%s\n", buffer_stable);
	
}

/**************************************************************
 Print inner states to default (within switch statements)
 **************************************************************/

// Update inwards (checking for AND and OR states). 
// Used when there is an outgoing transition from a superstate, so all its inner states must be
// updated to their default values
void printInnerstatesToDefault (node* state, node* transparent)
{
	while(state != NULL)
	{
		node* defaultChild = NULL;
		node* temptrans = NULL;
		node* tempchild = NULL;
		node* junct = juncList->head;

		// if the state is NOT a parent, same condition as (child->child == NULL)
		if (state->childId == 0) 
		{	
			if (state->rightSib != NULL) // rigthsib not NULL
				state = state->rightSib;
			else
			{	
				tempchild = state->parent;
				while ((tempchild->rightSib == NULL) && (tempchild != transparent) && (tempchild->parentId != 0)) {
					tempchild = tempchild->parent;
				}
				if (tempchild == transparent) state = NULL;
				else
					state = tempchild->rightSib;
			}
		}
		// state is a parent
		else
		{	
			// if the state is a parent
			//if(state->childId != 0)
			// Check if current state is AND-state. If current state is the one with execution order one, update!!
			// NOTE: We don't look up for the one with execution order one because we will check all inner states,
			// so if we update each time we find an AND state, we'll get multiple updates for the same AND state
			if (!strcmp(state->child->type, "AND_STATE"))
			{
				if (state->child->execOrder == 1)
				{	align();	
				printf("next(state%s) := state%s%s;\n",state->labelString, state->child->labelString,stateList->head->labelString);}
				
				state = state->child;
			}	
			// current state is OR-state
			else	
			{	
				// For OR_STATE without history:
				// If state is a parent, and it has a history junction, we don't need to update the state, 
				// (junct != NULL) & ((!strcmp(junct->type, "HISTORY_JUNCTION")) & (junct->parentId == state->parentId))
				// So we ask for the opposite condition in order to perform the update!
				// If history junctions exist, check if they are in the current state!
				if ((junct == NULL)  || ((!strcmp(junct->type, "HISTORY_JUNCTION")) && (junct->parentId != state->id)))
				{
					// find the child who has a transition with no src (is the default transition)
					tempchild = state->child;
					while (tempchild != NULL)	
					{
						temptrans = transList->head;
						while (temptrans != NULL)
						{
							if (temptrans->dst == tempchild->id && temptrans->src == 0)
							{	defaultChild = tempchild; break; }
							temptrans = temptrans->next;
						}
						tempchild = tempchild->rightSib;
					}
					
					align();	
					// There exist a default child
					if (defaultChild != NULL)
					{	
						printf("next(state%s) := state%s%s;\n", state->labelString, defaultChild->labelString,stateList->head->labelString); 
					}	
					// No default child is defined, direct transition leads to states within superstate
					else
					{	
						tempchild = state->child;
						while (tempchild != NULL)	
						{
							temptrans = transList->head;
							while (temptrans != NULL)
							{
								if ((temptrans->parent != NULL) && (temptrans->parent->parent->id != state->id))
								{	defaultChild = temptrans->parent; break; }
								temptrans = temptrans->next;
							}
							tempchild = tempchild->rightSib;
						}
						if (defaultChild != NULL)
						{	
							printf("next(state%s) := state%s%s;\n", state->labelString, defaultChild->labelString,stateList->head->labelString);
						}	
						else
						{	
							printf("next(state%s) := state%s%s;\n", state->labelString, state->child->labelString,stateList->head->labelString);
						}	
					}
										
				}
				// There is a junction in the current state. No updates needed, but update children appropriately
				state = state->child;
			}								
		}	
	}
}	

/**************************************************************
 Print outer states to current node value (within switch statements)
 **************************************************************/

// Update outwards in the hierarchy, up to the main chart 
// (checking for AND and OR states). 
void printOuterstatesToCurrent (node* temp, int *ANDupdating, int *ANDleveldone)
{
	while ((temp != NULL) && (temp->parent !=NULL))
	{
		
		// Check if parent is AND_STATE 
		if (!strcmp(temp->type, "AND_STATE"))
		{
			// Check if it's the innermost AND-state we are updating 
			if (*ANDupdating==ANDlevel)
				// Check if parent has a right sib to update to it (i.e., right sib is not NULL)
				if (temp->rightSib != NULL)
				{
					align(); printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->rightSib->labelString,stateList->head->labelString);
					*ANDleveldone=0;
				}
			// Check if parent is AND_STATE & parent doesn't have a right sib, so update to first AND_STATE in 
			// hierarchy (i.e., state with execution order 1)	
				else 
				{
					align(); printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->parent->child->labelString,stateList->head->labelString);
					*ANDleveldone=1;
				}	
			// Check up in the hierarchy
				else
					// The most immediate child AND-state has finished its ordered composition
					if (*ANDleveldone)
						// Check if parent has a right sib to update to it (i.e., right sib is not NULL)
						if (temp->rightSib != NULL)
						{
							align(); printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->rightSib->labelString,stateList->head->labelString);
							*ANDleveldone=0;
						}
						// Check if parent is AND_STATE & parent doesn't have a right sib, so update to first AND_STATE in 
						// hierarchy (i.e., state with execution order 1)	
						else 
						{
							align(); printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->parent->child->labelString,stateList->head->labelString);
							*ANDleveldone=1;
						}	
			// The most immediate child AND-state has NOT finished its ordered composition		
						else
						{	align(); printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->labelString,stateList->head->labelString);	}
			
			// Jan 13, 2009: Update sibling AND-states (and its substates) to the values they got last
			//!!!
			node* ANDsibling = NULL;
			node* ANDcurrent = temp;
			// Find sibling. If rightsib is not null, assign to it, otherwise, assign first sib
			if (ANDcurrent->rightSib != NULL)
				ANDsibling = ANDcurrent->rightSib;
			else 
				ANDsibling = ANDcurrent->parent->child;
			// update inwards
			while(ANDsibling != ANDcurrent)
			{
				// if the state is a parent
				if(ANDsibling->childId != 0)
				{
					align();	printf("next(state%s) := state%s;\n", ANDsibling->labelString, ANDsibling->labelString);
					node* statetmp = ANDsibling->child;
					while(statetmp != NULL)
					{
						// if the state is a parent
						if(statetmp->childId != 0)
						{
							align();	printf("next(state%s) := state%s;\n", statetmp->labelString, statetmp->labelString);
							statetmp = statetmp->child;
						}
						else // if state is not a parent, we still need to update all the inner states of siblings!
						{
							// find the child who has a transition with no src (is the default transition)
							node* tempstate = statetmp->rightSib;
							// If by the end of the cycle it's still NULL, it means none of the siblings have children, thus, no need 
							// to keep updating inwards! 
							statetmp = NULL; 
							while (tempstate != NULL)
							{
								if (tempstate->childId != 0)
								{	statetmp = tempstate; break; }
								tempstate = tempstate->rightSib;
							}
							tempstate = NULL;
						}
					}
					
				}
				else // if state is not a parent, we still need to update all the inner states of siblings!
				{
					align();	printf("next(state%s) := state%s;\n", ANDsibling->labelString, ANDsibling->labelString);
				}
				if (ANDsibling->rightSib != NULL)
					ANDsibling = ANDsibling->rightSib;
				else 
					ANDsibling = ANDsibling->parent->child;
				//state = state->next;
			}
			//!!!
		}	
		// OR-state
		else
		{ 
			align(); 
			printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->labelString,stateList->head->labelString);
		}
		temp = temp->parent;
	}	
}

/**************************************************************
 Print update of OR-state to its default value (in switch statements)
 **************************************************************/

void printORstateToDefault(node* state)
{
	node* junct = juncList->head;
	
	if ((junct == NULL) || ((!strcmp(junct->type, "HISTORY_JUNCTION")) && (junct->parentId != state->parentId))) 
	{	
		// find the child who has a transition with no src (is the default transition)
		node* temptrans = NULL;
		node* defaultChild = NULL;
		node* tempstate = state->parent->child;
		
		while (tempstate != NULL)	
		{
			temptrans = transList->head;
			while (temptrans != NULL)
			{
				if (temptrans->dst == tempstate->id && temptrans->src == 0)
				{	defaultChild = tempstate; break; }
				temptrans = temptrans->next;
			}
			tempstate = tempstate->rightSib;
		}
		
		// default transition for OR_STATE
		if (defaultChild != NULL) 
			// Exist default transition for OR-states
		{	align();	
			printf("next(state%s) := state%s%s;\n", state->parent->labelString, defaultChild->labelString,stateList->head->labelString); 
		}
		else
		{
			if (!strcmp(state->type, "OR_STATE"))
				// No default transition, only direct transition inwards (OR-states)
			{	align(); 
				printf("next(state%s) := state%s%s;\n", state->parent->labelString, state->parent->child->labelString,stateList->head->labelString); 
			}
			else
			{
				tempstate = state->parent->child;
				while (tempstate != NULL)	
				{
					if (!strcmp(tempstate->type, "AND_STATE") && tempstate->execOrder == 1)
					{	defaultChild = tempstate; break; }
					tempstate = tempstate->rightSib;
				}
				// default transition for AND_STATE (state with execution order 1)
				if (defaultChild != NULL)
				{ align();	printf("next(state%s) := state%s%s;\n", state->parent->labelString, defaultChild->labelString,stateList->head->labelString); }
			}
			defaultChild = NULL;
		}
	}	
}	

/**************************************************************
 Print update of AND-state to its next sibling value (in switch statements)
 **************************************************************/
void printANDstateToNextSib (node* state, int *ANDupdating, int *ANDleveldone)
{
	align();
	// Check if it's the state AND-state we are updating 
	if (*ANDupdating==ANDlevel)
	{	
		// Check if parent has a right sib to update to it (i.e., right sib is not NULL)
		if (state->rightSib != NULL)
		{
			printf("next(state%s) := state%s%s;\n", state->parent->labelString, state->rightSib->labelString,stateList->head->labelString);
			*ANDleveldone=0;
		}
		// Check if parent is AND_STATE & parent doesn't have a right sib, so update to first AND_STATE in 
		// hierarchy (i.e., state with execution order 1)	
		else 
		{
			node* defaultChild=NULL;
			node* tempstate = state->parent->child;
			while (tempstate != NULL)	
			{
				if (!strcmp(tempstate->type, "AND_STATE") && tempstate->execOrder == 1)
				{	defaultChild = tempstate; break; }
				tempstate = tempstate->rightSib;
			}
			// default transition for AND_STATE (state with execution order 1)
			if (defaultChild != NULL)
			{ align();	printf("next(state%s) := state%s%s;\n", state->parent->labelString, defaultChild->labelString,stateList->head->labelString); }			
			*ANDleveldone=1;
		}	
	}	
	// Check up in the hierarchy
	else
	{		
		// The most immediate child AND-state has finished its ordered composition
		if (*ANDleveldone)
		{	
			// Check if parent has a right sib to update to it (i.e., right sib is not NULL)
			if (state->rightSib != NULL)
			{
				printf("next(state%s) := state%s%s;\n", state->parent->labelString, state->rightSib->labelString,stateList->head->labelString);
				*ANDleveldone=0;
			}
		// Check if parent is AND_STATE & parent doesn't have a right sib, so update to first AND_STATE in 
		// hierarchy (i.e., state with execution order 1)	
			else 
			{
				node* defaultChild=NULL;
				node* tempstate = state->parent->child;
				while (tempstate != NULL)	
				{
					if (!strcmp(tempstate->type, "AND_STATE") && tempstate->execOrder == 1)
					{	defaultChild = tempstate; break; }
					tempstate = tempstate->rightSib;
				}
				// default transition for AND_STATE (state with execution order 1)
				if (defaultChild != NULL)
				{ align();	
				  printf("next(state%s) := state%s%s;\n", state->parent->labelString, defaultChild->labelString,stateList->head->labelString); }
				*ANDleveldone=1;
			}	
		}	
		// The most immediate child AND-state has NOT finished its ordered composition		
		else
			printf("next(state%s) := state%s%s;\n", state->parent->labelString, state->labelString,stateList->head->labelString);	
	}	
}	

/**************************************************************
Print the switch statements
**************************************************************/

// print the switch statement for the states in hierarchical fashion
// this function in recursive with a node with no children as the base case
void printSwitch(node* parent, int trans_hierarchy)
{

	node* child = parent->child;
	node* junct = NULL;
	node* trans = NULL;
	node* dataToPrint = NULL;
	int ANDupdating = 0; // flag that indicates when the upper levels can be updated
	int ANDleveldone = 0; // variable that holds the AND-state hierarchy level when updating transitions actions
	int Readyprinted1 = 0; // local boolean flag to check if the ready condition is already printed or not
	
	
	if (ANDstates==0)
	{
		if ((!Readyprinted1) && (!ReadyOR))
		{
			printf("if (sys_stable)\n");
			align(); printf("{\n");
			tabs++;
			align(); 
			Readyprinted1++; ReadyOR++;
		}
	}	
	else
	{
		if (parent->parentId != 0)
		{
			node* workingstate = parent->parent->child; //parent; //parent->parent->child
			node* innerAND = NULL; // variable to hold most inner AND-state with execution order 1
		
			// Check for the most inner AND-state with execution order 1
			while (workingstate != NULL)
			{
				if (!strcmp(workingstate->type, "AND_STATE")) 
					if (workingstate->execOrder == 1)
						if (workingstate->child != NULL)
							if (!strcmp(workingstate->child->type, "AND_STATE"))
								workingstate = workingstate->child;
							else
							{ innerAND = workingstate; workingstate = NULL; }
						else
						{ innerAND = workingstate; workingstate = NULL; }
					else
						workingstate = workingstate->rightSib; //next;
				else
					workingstate = NULL;
			}	
			// If (most inner AND-state with execution order 1) equal to myself (node that we are considering now = parent)
			// and if (child of an AND-state, parent has exec order 1) then print stable
			if (innerAND==parent) //((innerAND==parent) & (parentANDExec==0))
			{
				printf("if (sys_stable)\n");
				align(); printf("{\n");
				tabs++;
				align(); 
				Readyprinted1++; ReadyOR++;
			}
		}	
	}
	
	printf("switch(state%s)\n", parent->labelString);
	align(); printf("{\n");

	if (!strcmp(child->type, "AND_STATE") & (child->execOrder == 1))
		ANDlevel++;		

	tabs++;

	int order; // Variable to account for priority in transitions
	while(child != NULL)
	{
		printf("\n");
		align(); 
		printf("state%s%s : \n", child->labelString,stateList->head->labelString); 	
		tabs++; align();
		
		// find all transitions whos source is the child node
		trans = transList->head;
		order = 1;
		while(trans != NULL)
		{	
			// (trans->dst && execOrder=1) condition added in case of self-loops (inner-state transition)
			// Feb 11, 2011: Case of inner transition that is a self loop, print it first regardless or order!
			//((trans->src == child->id) && ((trans->execOrder == order) || (trans->dst == child->id)))
			// Apr 2, 2011: Make all transitions out self-loops, so no need to differenciate inner and
			// outer transitions
			if (((trans->src == child->id) && (trans->execOrder == order)) || 
				((trans->src == child->id) && (trans->dst == child->id) && (trans->execOrder == order))) 
				
				{
				char event[100] = "";
				char cond[100] = "";
				char action[500] = "";
				char condAction[500] = "";
				char CompData[100] = "";
				int eventsCheck = 0, condsCheck = 0, actionCheck = 0, condActionCheck = 0, eCheck = 0;
				
				eventsCheck = transEvent(trans, event);
				condsCheck = transCond(trans, cond);
				actionCheck = transAction(trans, action);
				condActionCheck = transCondAction(trans, condAction);
                eCheck = mkEventCond(stateList->head->labelString, event);    

				printf("if (");

				if (eventsCheck && eCheck)
					//printf("( %sevents = (%s) )", stateList->head->labelString, event);
                    printf("%s", event);
				
				if (eventsCheck && condsCheck)
					printf(" & ");
				
				if (!eventsCheck && !condsCheck)
					printf(" 1 ");	
					
				if (condsCheck)
					printf("( %s )", cond);

				printf(" )\n"); align(); printf("{\n");

				tabs++;	

				printf("\n");
				
				// Print the transition change for regular transitions
				// Transition labelled as "t#", where # is the id of the
				// associated transition (taken from the .mdl file)
				
				// Jan 31, 2011: Update hierarchically if necessary
				if (trans_hierarchy)
				{
					if (trans->parent != NULL)
					{	
						node* printstate = stateList->head;
						node* parentstate = trans->parent;
						
						// Feb 8, 2011: Find to which parent the trans is associated in the case it's not the most immediate
						// NOTE: New trans variable gets created at main chart level and immediately below AND-states;
						//		 OR-states get grouped if they are in the same branch.
						while (parentstate!=NULL) 
						{	
							if ((parentstate->parentId == 0) || ((parentstate->parentId != 0) && !strcmp(parentstate->type, "AND_STATE")))
								break;
							parentstate = parentstate->parent;
						}	
						
						while(printstate != NULL)
						{
							if (printstate->childId != 0) 
								if (((printstate->parentId == 0) && (!strcmp(printstate->child->type, "OR_STATE"))) || 
									((printstate->parentId != 0) && !strcmp(printstate->type, "AND_STATE")))	
									if (!strcmp(parentstate->labelString,printstate->labelString))
									{	align(); 
										printf("next(Trans%s) := t%d;\n", printstate->labelString, trans->id); 
									}
									else
									{	align(); 
										printf("next(Trans%s) := (sys_stable ? t0 : Trans%s);\n", printstate->labelString, 
														printstate->labelString); 
									}
							
							printstate = printstate->next;
						}
					}	
				}
				else	
				{	align(); printf("next(Trans%s) := t%d;\n", stateList->head->labelString, trans->id); 
				}	


				// print the outputs that AREN'T in the transitions
				// Create a string of the form "next(DataList->labelString)" to compare with the
				// transition label, so we can update the output data (to the previous value it had) if it is  
				// not already updated in actions or condition_actions (i.e., when the output data is updated
				// with the next() construction)... REASON: The output data can also be part of the actions or
				// condition actios if the output depends on other output, BUT it is not updated itself!!!
				dataToPrint = dataList->head;
				while (dataToPrint != NULL)
				{
					if (strstr(dataToPrint->scope, "OUTPUT") != NULL)
					{
						if(strstr(trans->labelString, dataToPrint->labelString) == NULL)
						{
							align(); 
							// If output is a variable associated to an actuator command, need to
							// check if the associated output variable was part of the action or condition
							// action of the transition. If so, assign explicit value, else, set value to 0.
							// CORRECTION (Aug 10, 2010): The values of all outputs must be retained,
							// including the actuator commands (set_act)!!! 
							// Otherwise, when it's time to check for the value (at stable), the
							// value for the actuator would be 0 even if it had changed in the macro-step
							// NOTE: what makes the difference is when the request boolean variable is set!
							printf("next(%s) := %s;\n", dataToPrint->labelString, dataToPrint->labelString);
						}
						else
						{
						
							strcpy(CompData,"next(");
							strcat(CompData,dataToPrint->labelString);
							strcat(CompData,")");
							CompData[strlen(CompData)]='\0';
							if((strstr(action, CompData) == NULL) && (strstr(condAction, CompData) == NULL))
							{
								align(); printf("next(%s) := %s;\n", dataToPrint->labelString, dataToPrint->labelString);
							}	
						}	
					}
					dataToPrint = dataToPrint->next;
				}


				// print the outputs that ARE in the transitions
				if (actionCheck)
				{
					align(); printf("%s\n", action);
				}
				if (condActionCheck)
				{
					align(); printf("%s\n", condAction);
				}
				
				// print the local data that AREN'T in the transitions
				dataToPrint = dataList->head;
				while (dataToPrint != NULL)
				{
					if (strstr(dataToPrint->scope, "LOCAL") != NULL)
					{
						if(strstr(trans->labelString, dataToPrint->labelString) == NULL)
						{
							align(); 
							// If local is a boolean variable associated to a parameterized command, need to
							// check if the associated output variable was part of the action or condition
							// action of the transition. If so, set value to 1, else, set value to 0.
							// Feb 16, 2011: Make boolean variable (_req) associated to parameterized commands
							// If trans requests a value on the actuator variable associated to the boolean event
							//		next(e) := 1
							// If trans doesn't request a value on the actuator associated to the boolean event
							//		next(e) := ~sys_stable & e
							// This approach allows us to avoid the use of macros to reset the value of the
							// event or the need to use history variables to use remainder semantics
							if (strstr(dataToPrint->labelString, "_req") != NULL) 
							{
								char buffer[500];
								char act_str[500];
								int i = 0; // for the labelString
								int j = 0; // for the buffer

								buffer[j++] = 'n';
								buffer[j++] = 'e';
								buffer[j++] = 'x';
								buffer[j++] = 't';
								buffer[j++] = '(';
								buffer[j++] = 's';
								buffer[j++] = 'e';
								buffer[j++] = 't';
								buffer[j++] = '_';
							
								while(dataToPrint->labelString[i] != '_')
								{
									buffer[j] = dataToPrint->labelString[i];
									act_str[i] = dataToPrint->labelString[i];
									i++;
									j++;
								}	
				
								buffer[j++] = ')';
								buffer[j] = '\0';
								act_str[i] = '\0';
								//printf("\t%s : NEW;\n", buffer);
			
								// Feb 21, 2011: Changes that allow us to have remainder semantics without the use of a macro or history
								if((strstr(action, buffer) != NULL) || (strstr(condAction, buffer) != NULL))
									printf("next(%s) := 1;\n", dataToPrint->labelString);
								else 
									printf("next(%s) := (~sys_stable & %s);\n", dataToPrint->labelString, dataToPrint->labelString);
							} else	
								// If local is not a boolean variable associated to a parameterized command, 
								// set new value to previous value.	
								printf("next(%s) := %s;\n", dataToPrint->labelString, dataToPrint->labelString);
						}
						else
						{
						
							strcpy(CompData,"next(");
							strcat(CompData,dataToPrint->labelString);
							strcat(CompData,")");
							CompData[strlen(CompData)]='\0';
							if((strstr(action, CompData) == NULL) && (strstr(condAction, CompData) == NULL))
							{
								align(); printf("next(%s) := %s;\n", dataToPrint->labelString, dataToPrint->labelString);
							}	
						}
						
					}
					dataToPrint = dataToPrint->next;
				}
				
				ANDupdating=ANDlevel;
				junct = juncList->head;

				// * Transition source AND destination within the same hierachy level *
				// if the transition stays within the same parent superstate then update the parent state with the active state.
				// (In the first "next(state%s):=%s;"). Make the temp equal to parent, and update the hierachy using the parents.
				// Also, update all parent states in the hierarchy: If OR_STATE, parent of a parent is updated with parent's name
				// If AND_STATE, parent of a parent is updated with right_sib or first sib (depending on rotation for AND_STATES)
				// NOTE (as of results of FI analysis from Aug 20, 2008): Need to update substates of superstates for outgoing
				// transitions to default states, because otherwise we are acting as having history state!!
				// NOTE (as of results of FI analysis with YZ in Dec01_2010): Need to update substates for outgoing trans, including
				//	inner AND states (not only OR-states)
				// NOTE (Feb 8, 2011): Check if both src and dst are in the same hierarchy level. If so, if src state is a superstate
				// then update itsef and update inwards, otherwise just update itself	
				if ((trans->parent->parentId == parent->id) && (trans->child->parentId == parent->id))
				{	
					printf("\n");
					//align();
					
					// NOTE (Sep 6, 2008):
					// Check if src of transition is a superstate. If so, we need to update any of its substates (even hierarchicaly) 
					// to their default states!! Before, we had assumed that if the transition destination was in the same hierarchy
					// level, the destination state would be within the hierarchy level as the source state in the previous step, but
					// this is not true given that the Stateflow semantics give higher priority to outoing transitions than to internal
					// transitions, and for outgoing transitions, Stateflow gives higher priority to transitions from the most external 
					// states in the hierarchy. It is also important to note that, in the previous final version (Apr 2008), we were
					// updating only the state variables at the level of hierarchy where the transition was occuring, so at the high 
					// levels of hierarchy, the state variables at lower level of hierarchy were not being updated, which made those
					// superstates at lower level of hierarchy act as if they had a history state (error in modelling). Now, at any
					// level of hierarchy, we need to update all hierarchy levels to their correct values to avoid the modelling error.
										
					
					// (1) If trans source is a parent state (superstate), so need to update all its substates (update inwards)
					// Feb 8, 2011: If parent is not superstate, no need to update inwards and no need to update siblings.
					if (trans->parent->child != NULL) 
						printInnerstatesToDefault(trans->parent, trans->parent);
					
					// (2) Update myself. NOTE if trans has src and dst, they must be OR-states (No transitions can be src or dst of AND-states)
					align();
					printf("next(state%s) := state%s%s;\n", parent->labelString, trans->child->labelString,stateList->head->labelString);
					
					// (3) Update up in the hierarchy up to the main chart
					node* temp = parent;
					printOuterstatesToCurrent (temp, &ANDupdating, &ANDleveldone);
					//ANDleveldone=0;
				}
				
				// * Transition destination NOT in the same hierachy level *
				// When the transition src and dst are not in the same hierarchy level:
				// (1) Find the commonParent of both ends.
				// (2) Then update each transition end up to the commonParent (updating at each steps to the parent of the current node):
					// (2a) For outermost=trans->parent end, update parent node to its initial state
					// (2b) For innermost=trans->child end, update parent node to the nodes current label
				// (3) Then update from commonParent to the main chart.
					// First, update commonParent with the outermost node label
					// Second, update parent node to its current node label
				// (4) Finally, check if trans->parent has children. If so, update inwards to default values  	
				else
				{
					printf("\n");
					
					// (4) If trans->parent has children, update inwards to default values  
					if (trans->parent->child != NULL) 
						printInnerstatesToDefault(trans->parent, trans->parent);
					
					node* innermost = trans->child;
					node* outermost = trans->parent;
					node* commonParent = NULL;
					
					// (1) Find common parent
					while (commonParent == NULL) 
					{
						// Check if trans->child's end has reached the main chart level
						// If so, update ends and check again
						if (innermost->parent == NULL)
						{
							if (outermost->parent == NULL) break;
							else
							{
								innermost = trans->child;
								outermost = outermost->parent;
							}								
						}	
						// Check if both ends have a common parent
						else
						{	
							if (innermost->parent->id == outermost->parent->id) 
								commonParent = innermost->parent;
							innermost = innermost->parent;
						}	
					}
					
					// (2) Then update each transition end up to the commonParent
					if (commonParent!=NULL)
					{
						// (2a) Update outermost (trans->parent end) => going "outwards", update to defaults
						outermost = trans->parent;
						while ((outermost->parent != commonParent) && (outermost->parent !=NULL))
						{
							if (!strcmp(outermost->type, "AND_STATE"))
							{
								// Check if it's the state AND-state we are updating 
								if (ANDupdating==ANDlevel)
								{	
									node* defaultChild=NULL;
									node* tempstate = outermost->parent->child;
									while (tempstate != NULL)	
									{
										if (!strcmp(tempstate->type, "AND_STATE") && tempstate->execOrder == 1)
										{	defaultChild = tempstate; break; }
										tempstate = tempstate->rightSib;
									}
									// default transition for AND_STATE (state with execution order 1)
									if (defaultChild != NULL)
									{ align();	
									  printf("next(state%s) := state%s%s;\n", outermost->parent->labelString, defaultChild->labelString,stateList->head->labelString); }										
									ANDleveldone=1;
								}	
								// Check up in the hierarchy
								else
								{		
									// The most immediate child AND-state has finished its ordered composition
									if (ANDleveldone)
									{	
										node* defaultChild=NULL;
										node* tempstate = outermost->parent->child;
										while (tempstate != NULL)	
										{
											if (!strcmp(tempstate->type, "AND_STATE") && tempstate->execOrder == 1)
											{	defaultChild = tempstate; break; }
											tempstate = tempstate->rightSib;
										}
										// default transition for AND_STATE (state with execution order 1)
										if (defaultChild != NULL)
										{ align();	
										printf("next(state%s) := state%s%s;\n", outermost->parent->labelString, defaultChild->labelString,stateList->head->labelString); }										
										ANDleveldone=1;	
									}	
									// The most immediate child AND-state has NOT finished its ordered composition		
									else
										printf("next(state%s) := state%s%s;\n", outermost->parent->labelString, outermost->labelString,stateList->head->labelString);	
									ANDupdating--;
								}	
							}	
							// OR state, updating outwards
							else
							{
								// update to the default transition if history junction doesn't exist, or the current hierarchy level doens't have history
								if (junct == NULL)	  
								{	printORstateToDefault(outermost);	}	
								else
									if ((!strcmp(junct->type, "HISTORY_JUNCTION")) && (junct->parentId != innermost->parentId))
									{	printORstateToDefault(outermost);	}
							}
							outermost = outermost->parent;
						}	
						
						// (2b) Update innermost (trans->child end) => going "inwards", update to current value
						innermost = trans->child;
						while ((innermost->parent != commonParent) && (innermost->parent !=NULL))
						{
							if (!strcmp(innermost->type, "AND_STATE"))
							{
								printANDstateToNextSib(innermost, &ANDupdating, &ANDleveldone);	
							}	
							// OR state, updating inwards
							else
							{
								// If trans->child is dst, update even if the dst state is in a state with history!
								if (innermost->id == trans->child->id)
								{ align();	
								  printf("next(state%s) := state%s%s;\n", innermost->parent->labelString, innermost->labelString,stateList->head->labelString);   
								}
								else
								{	
									// update to the node's current value if history junction doesn't exist, or the current hierarchy level doens't have history
									if (junct == NULL) 
									{ align();	
									  printf("next(state%s) := state%s%s;\n", innermost->parent->labelString, innermost->labelString,stateList->head->labelString);  
									}
									else
									{	
										if ((!strcmp(junct->type, "HISTORY_JUNCTION")) && (junct->parentId != innermost->parentId))
										{ align();	
										  printf("next(state%s) := state%s%s;\n", innermost->parent->labelString, innermost->labelString,stateList->head->labelString);  
										}	
									}
								}	
							}
							innermost = innermost->parent;
						}
						
						// (3) Update from commonParent up to the main chart to the current's node value
						align();
						// update commonParent
						if (commonParent == trans->child->parent)
						{	
							printf("next(state%s) := state%s%s;\n", commonParent->labelString, trans->child->labelString,stateList->head->labelString);
						}
						else
							if (commonParent == trans->child->parent)
							{	
								printf("next(state%s) := state%s%s;\n", commonParent->labelString, trans->parent->labelString,stateList->head->labelString);
							}
							else
							{	
								printf("next(state%s) := state%s%s;\n", commonParent->labelString, innermost->labelString,stateList->head->labelString);
							}	
						// update from commonParent up
						while (commonParent->parent != NULL)
						{
							if (!strcmp(commonParent->type, "AND_STATE"))
							{
								printANDstateToNextSib(commonParent, &ANDupdating, &ANDleveldone);	
							}	
							// OR state, updating inwards
							else
							{
								// update to the node's current value if history junction doesn't exist, or the current hierarchy level doens't have history
								if (junct == NULL) 
								{ align();	
								  printf("next(state%s) := state%s%s;\n", commonParent->parent->labelString, commonParent->labelString,stateList->head->labelString);  
								}
								else
									if ((!strcmp(junct->type, "HISTORY_JUNCTION")) && (junct->parentId != commonParent->parentId))
									{ align();	
									  printf("next(state%s) := state%s%s;\n", commonParent->parent->labelString, commonParent->labelString,stateList->head->labelString);
									}	
							}
							commonParent = commonParent->parent;
						}
						
					}	
					
					ANDleveldone=0;	
										
				}
			
					
				tabs--;
				align(); 
				printf("}\n"); align(); printf("else ");
				
				// In case of inner transition. Don't increase order of execution and check rest of trans	
				/*if (trans->dst == child->id)
					trans = trans->next;
				else 
				{	
					order++;
					// Start from the beginning of the list in case transitions are out of order! 
					trans = transList->head;
				}*/	
				order++;
				// Start from the beginning of the list in case transitions are out of order! 
				trans = transList->head;	
			}
			else
			 trans = trans->next;
		}
		
		

		printf("\n"); align(); printf("{\n");
		tabs++;
		
		
		
		// stop printing switch statements when there are no more superstates
		if(child->child != NULL)
		{
		
			align(); printSwitch(child, trans_hierarchy);
			
		}
		// SELF-LOOPS!!!
		// self-loops, making outputs persist.
		else
		{
			printf("\n");
			// Print the transition macro change for looping transitions
			// Transition labelled as "tn" for do-nothing transitions
			
			// Jan 31, 2011: Update hierarchically if necessary
			if (trans_hierarchy)
			{
				node* printstate = stateList->head;
				node* parentstate = child->parent;
				
				// Feb 8, 2011: Find to which parent the trans is associated in the case it's not the most immediate
				// NOTE: New trans variable gets created at main chart level and immediately below AND-states;
				//		 OR-states get grouped if they are in the same branch.
				while (parentstate!=NULL) 
				{	
					if (((parentstate->parentId == 0) && (!strcmp(parentstate->child->type, "OR_STATE"))) || 
						((parentstate->parentId != 0) && !strcmp(parentstate->type, "AND_STATE")))
						break;
					parentstate = parentstate->parent;
				}	
				
				// Initialize hierarchy of transitions if it exist
				while(printstate != NULL)
				{
					if (printstate->childId != 0) 
						if (((printstate->parentId == 0) && (!strcmp(printstate->child->type, "OR_STATE"))) || 
							((printstate->parentId != 0) && !strcmp(printstate->type, "AND_STATE")))	
							if (!strcmp(parentstate->labelString,printstate->labelString))
							{	align(); 
								printf("next(Trans%s) := tn;\n", printstate->labelString); 
							}
							else
							{	align(); 
								printf("next(Trans%s) := (sys_stable ? t0 : Trans%s);\n", printstate->labelString, 
												printstate->labelString); 
							}
					
					printstate = printstate->next;
				}
			}
			else
			{	align(); printf("next(Trans%s) := tn;\n", stateList->head->labelString); 
			}
			
			dataToPrint = dataList->head;
			while (dataToPrint != NULL)
			{
				if (strstr(dataToPrint->scope, "OUTPUT") || strstr(dataToPrint->scope, "LOCAL_DATA") != NULL)
				{
					align();
					// If local is a boolean variable associated to a parameterized command, set value to 0.
					// Feb 16, 2011: Make boolean variable (_req) associated to parameterized commands
					// If trans doesn't request a value on the actuator associated to the boolean event
					//		next(e) := ~sys_stable & e
					// This approach allows us to avoid the use of macros to reset the value of the
					// event or the need to use history variables to use remainder semantics
					if (strstr(dataToPrint->labelString, "_req") != NULL) 
						printf("next(%s) := (~sys_stable & %s);\n", dataToPrint->labelString, dataToPrint->labelString);
					else
							// CORRECTION (Aug 10, 2010): The values of all outputs must be retained,
							// including the actuator commands (set_act)!!!
							// Otherwise, when it's time to check for the value (at stable), the
							// value for the actuator would be 0 even if it had changed in the macro-step
							// NOTE: The _req variable will define when the command actually takes place
							printf("next(%s) := %s;\n", dataToPrint->labelString, dataToPrint->labelString);
							/*if(strstr(dataToPrint->labelString, "set_") != NULL)
							 {
								printf("next(%s) := %d;\n", dataToPrint->labelString, 0);
							 }
							 else
								printf("next(%s) := %s;\n", dataToPrint->labelString, dataToPrint->labelString);*/
						
				}
				dataToPrint = dataToPrint->next;
				
			}
			
			printf("\n");
			
			ANDupdating=ANDlevel;
			
			// * Transition destination within the same hierachy level *
			// if the transition stays within the same parent superstate then update the parents state with the active state
			// Also, update all parent states in the hierarchy: If OR_STATE, parent of a parent is updated with parent's name
			// If AND_STATE, parent of a parent is updated with right_sib or first sib (depending on rotation for AND_STATES)
			// Note Jan 6, 2009: Need to update up in the hierarchy, but also down in the hierarchy, as well as to find out
			// if siblings are superstates and update inner hierarchies.
			node* temp = child;

			align();
			if (!strcmp(temp->type, "AND_STATE"))
				{
					// Check if it's the innermost AND-state we are updating 
					if (ANDupdating==ANDlevel)
						// Check if parent has a right sib to update to it (i.e., right sib is not NULL)
						if (temp->rightSib != NULL)
						{
							printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->rightSib->labelString,stateList->head->labelString);
							ANDleveldone=0;
						}
						// Check if parent is AND_STATE & parent doesn't have a right sib, so update to first AND_STATE in 
						// hierarchy (i.e., state with execution order 1)	
						else 
						{
							printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->parent->child->labelString,stateList->head->labelString);
							ANDleveldone=1;
						}	
					// Check up in the hierarchy
					else
						// The most immediate child AND-state has finished its ordered composition
						if (ANDleveldone)
							// Check if parent has a right sib to update to it (i.e., right sib is not NULL)
							if (temp->rightSib != NULL)
							{
								printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->rightSib->labelString,stateList->head->labelString);
								ANDleveldone=0;
							}
							// Check if parent is AND_STATE & parent doesn't have a right sib, so update to first AND_STATE in 
							// hierarchy (i.e., state with execution order 1)	
							else 
							{
								printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->parent->child->labelString,stateList->head->labelString);
								ANDleveldone=1;
							}	
						// The most immediate child AND-state has NOT finished its ordered composition		
						else
						{
							printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->labelString,stateList->head->labelString);	
						}	
					ANDupdating--;
					
				}			
				else
				{	
					// OR_STATE, else self-loop statements -> No need to deal with history 
					printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->labelString,stateList->head->labelString);
				}	
				
				// NOTE (Jan 6, 2009):
				// Check if state is a superstate. If so, we need to update any of its substates (even hierarchicaly) 
				// to their default states!! Before, we were
				// updating only the state variables at the level of hierarchy where the self-loop was occuring, but 
				// we also need to find out if siblings are superstates and update inner hierarchies.
				
				/// node* temp = child;
				
				// printed the transition that was being taken at the current hierarchy level, and need to choose state in the 
				// next inwards hierarchy level (if any)
				node* state = NULL;
				// State is a superstate, so need to update all its substates
				if (temp->child != NULL) 
					state = temp->child;
				// Find if siblings are superstates!
				else
					{
						node* tempstate = temp->parent->child;
						int currentstateID = temp->id;
						// If by the end of the cycle state is still NULL, it means none of the siblings have children, thus, no need 
						// to keep updating inwards! 
						while (tempstate != NULL)
						{
							if ((tempstate->childId != 0) && (tempstate->id != currentstateID))
							{	state = tempstate; break; }
							tempstate = tempstate->rightSib;
						}
						tempstate = NULL;
						
					}
				
				// update inwards
				node* defaultChild = NULL;
				node* temptrans = NULL;
				node* tempchild = NULL;
				while(state != NULL)
				{
					// if the state is a parent
					if(state->childId != 0)
					{
						// For OR_STATE with history:
						// If state is a parent, and it has a history junction, we don't need to update the state, 
						// (junct != NULL) & ((!strcmp(junct->type, "HISTORY_JUNCTION")) & (junct->parentId == state->parentId))
						// So we ask for the opposite condition in order to perform the update!
						if (junct == NULL)  
						{
							// find the child who has a transition with no src (is the default transition)
							tempchild = state->child;
							while (tempchild != NULL)	
							{
								temptrans = transList->head;
								while (temptrans != NULL)
								{
									if (temptrans->dst == tempchild->id && temptrans->src == 0)
									{	defaultChild = tempchild; break; }
									temptrans = temptrans->next;
								}
								tempchild = tempchild->rightSib;
							}
							
							// default transition for OR_STATE
							if (defaultChild != NULL)
							{ align();	
							  printf("next(state%s) := state%s%s;\n", state->labelString, defaultChild->labelString,stateList->head->labelString);
							}
							else
							{
								tempchild = state->child;
								while (tempchild != NULL)	
								{
									if (!strcmp(tempchild->type, "AND_STATE") && tempchild->execOrder == 1)
									{	defaultChild = tempchild; break; }
									tempchild = tempchild->rightSib;
								}
								// default transition for AND_STATE (state with execution order 1)
								if (defaultChild != NULL)
								{ align();	printf("next(state%s) := state%s%s;\n", state->labelString, defaultChild->labelString,stateList->head->labelString);}
							}
							defaultChild = NULL;
						}
						else // There's junction but not in the parent state!
							if ((!strcmp(junct->type, "HISTORY_JUNCTION")) && (junct->parentId != state->parentId))
							{
								// find the child who has a transition with no src (is the default transition)
								tempchild = state->child;
								while (tempchild != NULL)	
								{
									temptrans = transList->head;
									while (temptrans != NULL)
									{
										if (temptrans->dst == tempchild->id && temptrans->src == 0)
										{	defaultChild = tempchild; break; }
										temptrans = temptrans->next;
									}
									tempchild = tempchild->rightSib;
								}
								
								// default transition for OR_STATE
								if (defaultChild != NULL)
								{ align();	
								  printf("next(state%s) := state%s%s;\n", state->labelString, defaultChild->labelString,stateList->head->labelString);  
								}
								else
								{
									tempchild = state->child;
									while (tempchild != NULL)	
									{
										if (!strcmp(tempchild->type, "AND_STATE") && tempchild->execOrder == 1)
										{	defaultChild = tempchild; break; }
										tempchild = tempchild->rightSib;
									}
									// default transition for AND_STATE (state with execution order 1)
									if (defaultChild != NULL)
									{ align();	printf("next(state%s) := state%s%s;\n", state->labelString, defaultChild->labelString,stateList->head->labelString);}
								}
								defaultChild = NULL;
							}
						node* tempstate = state->child;
						// If by the end of the cycle it's still NULL, it means none of the siblings have children, thus, no need 
						// to keep updating inwards! 
						state = NULL; 
						while (tempstate != NULL)
						{
							if (tempstate->childId != 0)
							{	state = tempstate; break; }
							tempstate = tempstate->rightSib;
						}
						tempstate = NULL;
						
					}
					else // if state is not a parent, we still need to find default transition for all the inner states!
					{
						// find the child who has a transition with no src (is the default transition)
						tempchild = state;
						while (tempchild != NULL)	
						{
							temptrans = transList->head;
							while (temptrans != NULL)
							{
								if (temptrans->dst == tempchild->id && temptrans->src == 0)
								{	defaultChild = tempchild; break; }
								temptrans = temptrans->next;
							}
							tempchild = tempchild->rightSib;
						}
						
						// default transition for OR_STATE
						if (defaultChild != NULL)
						{ align();	
						  printf("next(state%s) := state%s%s;\n", state->parent->labelString, defaultChild->labelString,stateList->head->labelString);    
						}
						else
						{
							tempchild = state->child;
							while (tempchild != NULL)	
							{
								if (!strcmp(tempchild->type, "AND_STATE") && tempchild->execOrder == 1)
								{	defaultChild = tempchild; break; }
								tempchild = tempchild->rightSib;
							}
							// default transition for AND_STATE (state with execution order 1)
							if (defaultChild != NULL)
							{ align();	printf("next(state%s) := state%s%s;\n", state->parent->labelString, defaultChild->labelString,stateList->head->labelString);}
						}
						defaultChild = NULL;
						node* tempstate = state->rightSib;
						// If by the end of the cycle it's still NULL, it means none of the siblings have children, thus, no need 
						// to keep updating inwards! 
						state = NULL; 
						while (tempstate != NULL)
						{
							if (tempstate->childId != 0)
							{	state = tempstate; break; }
							tempstate = tempstate->rightSib;
						}
						tempstate = NULL;
					}
					//state = state->next;
				}
				
				// Update up in the hierarchy
				temp = child->parent;
				while ((temp != NULL) && (temp->parent !=NULL))
				{
					align(); 
					// Check if parent is AND_STATE 
					if (!strcmp(temp->type, "AND_STATE"))
					{
						// Check if it's the innermost AND-state we are updating 
						if (ANDupdating==ANDlevel)
							// Check if parent has a right sib to update to it (i.e., right sib is not NULL)
							if (temp->rightSib != NULL)
							{
								printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->rightSib->labelString,stateList->head->labelString);
								ANDleveldone=0;
							}
						// Check if parent is AND_STATE & parent doesn't have a right sib, so update to first AND_STATE in 
						// hierarchy (i.e., state with execution order 1)	
							else 
							{
								printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->parent->child->labelString,stateList->head->labelString);
								ANDleveldone=1;
							}	
						// Check up in the hierarchy
							else
								// The most immediate child AND-state has finished its ordered composition
								if (ANDleveldone)
									// Check if parent has a right sib to update to it (i.e., right sib is not NULL)
									if (temp->rightSib != NULL)
									{
										printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->rightSib->labelString,stateList->head->labelString);
										ANDleveldone=0;
									}
						// Check if parent is AND_STATE & parent doesn't have a right sib, so update to first AND_STATE in 
						// hierarchy (i.e., state with execution order 1)	
									else 
									{
										printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->parent->child->labelString,stateList->head->labelString);
										ANDleveldone=1;
									}	
						// The most immediate child AND-state has NOT finished its ordered composition		
									else
										printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->labelString,stateList->head->labelString);	
						ANDupdating--;
						
						// Jan 13, 2009: Update sibling AND-states (and its substates) to the values they got last
						//!!!
						node* ANDsibling = NULL;
						node* ANDcurrent = temp;
						// Find sibling. If rightsib is not null, assign to it, otherwise, assign first sib
						if (ANDcurrent->rightSib != NULL)
							ANDsibling = ANDcurrent->rightSib;
						else 
							ANDsibling = ANDcurrent->parent->child;
						// update inwards
						while(ANDsibling != ANDcurrent)
						{
							// if the state is a parent
							if(ANDsibling->childId != 0)
							{
								align();	printf("next(state%s) := state%s;\n", ANDsibling->labelString, ANDsibling->labelString);
								node* statetmp = ANDsibling->child;
								while(statetmp != NULL)
								{
									// if the state is a parent
									if(statetmp->childId != 0)
									{
										align();	printf("next(state%s) := state%s;\n", statetmp->labelString, statetmp->labelString);
										statetmp = statetmp->child;
									}
									else // if state is not a parent, we still need to update all the inner states of siblings!
									{
										// find the child who has a transition with no src (is the default transition)
										node* tempstate = statetmp->rightSib;
										// If by the end of the cycle it's still NULL, it means none of the siblings have children, thus, no need 
										// to keep updating inwards! 
										statetmp = NULL; 
										while (tempstate != NULL)
										{
											if (tempstate->childId != 0)
											{	statetmp = tempstate; break; }
											tempstate = tempstate->rightSib;
										}
										tempstate = NULL;
									}
								}
								
							}
							else // if state is not a parent, we still need to update all the inner states of siblings!
							{
								align();	printf("next(state%s) := state%s;\n", ANDsibling->labelString, ANDsibling->labelString);
							}
							if (ANDsibling->rightSib != NULL)
								ANDsibling = ANDsibling->rightSib;
							else 
								ANDsibling = ANDsibling->parent->child;
							//state = state->next;
						}
						//!!!
					}
					// OR-state
					else
					{	
						printf("next(state%s) := state%s%s;\n", temp->parent->labelString, temp->labelString,stateList->head->labelString);
					}
					temp = temp->parent;
				}
				
				
			
			ANDleveldone=0;	
			
		}
		tabs--;		
		align(); printf("}");
		tabs--;
		child = child->rightSib;
	}


	tabs--;
	

	printf("\n");
	align(); printf("}\n");
	
	
	////////
	if (Readyprinted1)
	{
		tabs--;
		printf("\n");
		align(); printf("}\n");
		align(); printf("else\n");
		align(); printf("{\n");
		tabs++;
			
		// Print the transition macro change for looping transitions
		// Updated to last value
		
		// Jan 31, 2011: Update hierarchically if necessary
		if (trans_hierarchy)
		{
			node* printstate = stateList->head;
			// Update hierarchy of transitions if it exist
			while(printstate != NULL)
			{
				if (printstate->childId != 0) 
					if (((printstate->parentId == 0) && (!strcmp(printstate->child->type, "OR_STATE"))) || 
						((printstate->parentId != 0) && !strcmp(printstate->type, "AND_STATE")))	
					{	align(); 
						printf("next(Trans%s) := Trans%s;\n", printstate->labelString,printstate->labelString); 
					}
				
				printstate = printstate->next;
			}
		}
		else
		{	align(); printf("next(Trans%s) := Trans%s;\n", stateList->head->labelString, stateList->head->labelString); 
		}
		
		// print the outputs that AREN'T in the transitions
		dataToPrint = dataList->head;
		while (dataToPrint != NULL)
		{
			if (strstr(dataToPrint->scope, "OUTPUT") != NULL)
			{
				align(); 
				// If output variable associated to an actuator command, set value to 0.
				// (avoid multiple values during M/C). Else, set new value to previous value.
				// CORRECTION (Aug 10, 2010): The values of all outputs must be retained
				// when the feature idles, including the actuator commands (set_act)!!!
				// Otherwise, when it's time to check for the value (at stable), the
				// value for the actuator would be 0 even if it had changed in the macro-step
				printf("next(%s) := %s;\n", dataToPrint->labelString, dataToPrint->labelString);
				/*if(strstr(dataToPrint->labelString, "set_") != NULL)
				{
					printf("next(%s) := %d;\n", dataToPrint->labelString, 0);
				}
				else
					printf("next(%s) := %s;\n", dataToPrint->labelString, dataToPrint->labelString);*/
			}
			dataToPrint = dataToPrint->next;
		}

		// print the local data that AREN'T in the transitions
		dataToPrint = dataList->head;
		while (dataToPrint != NULL)
		{
			if (strstr(dataToPrint->scope, "LOCAL") != NULL)
			{
				align(); 
				// If local is a boolean variable associated to a parameterized command, set value to 0.
				// Else, set new value to previous value.
				// Feb 16, 2011: Make boolean variable (_req) associated to parameterized commands
				// If trans requests a value on the actuator variable associated to the boolean event
				//		next(e) := 1
				// If trans doesn't request a value on the actuator associated to the boolean event
				//		next(e) := ~sys_stable & e
				// This approach allows us to avoid the use of macros to reset the value of the
				// event or the need to use history variables to use remainder semantics
				if (strstr(dataToPrint->labelString, "_req") != NULL) 
					printf("next(%s) := (~sys_stable & %s);\n", dataToPrint->labelString, dataToPrint->labelString);
				else
					printf("next(%s) := %s;\n", dataToPrint->labelString, dataToPrint->labelString);
			}
			dataToPrint = dataToPrint->next;
		}
			
		printf("\n");
		
		// Jan 14, 2009: Updated to account for all states in all hierarchy levels 
		// In here, since it's the else to hold up into from all variables, we need
		// to update states from the stateheadList, in all hierarchy levels.
		// Added code delimited by comments //!!!
		//!!!
		node* state = stateList->head;
		while(state != NULL)
		{
			// if the state is a parent
			if(state->childId != 0)
			{
				align();	printf("next(state%s) := state%s;\n", state->labelString, state->labelString);
			}
			state = state->next;
		}
		//!!!
		
		Readyprinted1--;
		tabs--;
		align(); printf("}\n");
		tabs++;
				
	}	
			
	
	if (parent==stateTree->head)
		printStable(stateList->head);
		

			
}



/**************************************************************
 Main print function call
 **************************************************************/
void printSMV(int trans_hierarchy)
{
	int parms = 0;
	printf("MODULE ");
	node* temp = stateList->head;
	
	printf("%s (", temp->labelString);
	
	// find if we have any non-local events
	temp = eventList->head;
	while (temp != NULL)
	{
		// if str str doesn't find "LOCAL" it returns NULL
		if(strstr(temp->scope, "LOCAL") == NULL)
			parms++;
		temp = temp->next;
	}
	// if we found non-local events
	if(parms)
		printf("%sevents", stateList->head->labelString);
	
	temp = dataList->head;
	while (temp != NULL)
	{
		// if str str finds "INPUT" it doesn't returns NULL
		if(strstr(temp->scope, "INPUT") != NULL)
		{
			if (!parms)
				printf("%s", temp->labelString);
			else
				printf(",%s", temp->labelString);
			parms++;
		}
		
		temp = temp->next;
	}
	temp = dataList->head;
	while (temp != NULL)
	{
		// if str str finds "OUTPUT" it doesn't returns NULL
		if(strstr(temp->scope, "OUTPUT") != NULL)
		{
			if (!parms)
				printf("%s", temp->labelString);
			else
				printf(",%s", temp->labelString);
			parms++;
		}
		
		temp = temp->next;
	}
	
	printf(")\n{\n");
	tabs++;
	
	printInputOutput();
	printStates();
	printTrans(trans_hierarchy);
	printInits(trans_hierarchy);
	align();
	printSwitch(stateTree->head, trans_hierarchy);
	
	tabs--;
	printf("\n\n} \n");
	
}



/*****************************************************************************************
*****************     TRANSITION LABEL PARSING FUNCTIONS     *****************************
*****************************************************************************************/

/**************************************************************
FIND EVENT IN TRANSIION LABEL
**************************************************************/

// function to copy the event of a transition to the string given as a parameter
// returns 1 if there is an event in the transition, returns 0 is there is no event or an ERROR occurs
int transEvent(node* trans, char* event)
{
	char buffer[100];
	int i = 0; // index for the labelString and buffer
	
	if(trans == NULL || trans->labelString == NULL)
		return 0;

	while(trans->labelString[i] != '[' && trans->labelString[i] != '{' && trans->labelString[i] != '/' && trans->labelString[i] != '\0')
	{
		buffer[i] = trans->labelString[i];
		i++;
	}

	// if we never entered the while loop, ie there was no event, return NULL
	if (i == 0)
		return 0;

	// else we did find a condition so cap off the string, cpy the buffer and return 1
	else
	{
		buffer[i] = '\0';
		strcpy(event, buffer);
		return 1;
	}
}


/**************************************************************
FIND CONDITION IN TRANSIION LABEL
**************************************************************/

int transCond(node* trans, char* cond)
{
	char buffer[100];
	int i = 0; // index for the labelString
	int j = 0; // index for the buffer
	
	if(trans == NULL || trans->labelString == NULL)
		return 0;

	// loop for the whole string until we hit the '[' character
	while(trans->labelString[i] != '[')
	{
		// if we hit a null character there is no condition
		if(trans->labelString[i] == '\0')
			return 0;

		// else if the character wasn't a '[' then go to the next character and check that
		else
			i++;

	}

	// increment the labelString index so we're not still on the '[' character
	i++; 

	// add the characters to the buffer until we find the closing bracket
	while(trans->labelString[i] != ']')
	{
		if (trans->labelString[i] == '!')
		{
			buffer[j] = '~';
			i++;
			j++;
		}
		else if (	( trans->labelString[i] == '&' && trans->labelString[i+1] == '&' )	||
				( trans->labelString[i] == '|' && trans->labelString[i+1] == '|' )	||
				( trans->labelString[i] == '=' && trans->labelString[i+1] == '=' )	)
		{
			buffer[j] = trans->labelString[i];
			i+=2;
			j++;
		}
		else if ( (trans->labelString[i] == '<' || trans->labelString[i] == '>') && trans->labelString[i+1] != '=')
		{
			buffer[j++] = ' ';
			buffer[j++] = trans->labelString[i++];
			buffer[j++] = ' ';
		}

		else
		{
			buffer[j] = trans->labelString[i];
			i++;
			j++;
		}
	}
	
	// we found a condition so cap off the string, copy the buffer and return 1
	buffer[j] = '\0';
	strcpy(cond, buffer);
	return 1;


}


/**************************************************************
FIND TRANSITION ACTIONS IN TRANSIION LABEL 
**************************************************************/

int transAction(node* trans, char* action)
{
	
	char buffer[500];
	int i = 0; // index for the labelString
	int j = 0; // index for the buffer
	
	// error catching
	if(trans == NULL || trans->labelString == NULL)
		return 0;

	// loop for the whole string until we hit a '/' character indicating a transition action
	while(trans->labelString[i] != '/')
	{
		// if we hit a null character there is no transition actions
		if(trans->labelString[i] == '\0')
			return 0;

		// else if the character wasn't a '/' then go to the next character and check that
		else
			i++;
	}

	// move the index past the '/' that we found and discard it
	i++;


	// if we got here then we have actions so loop for the rest of the string
	while(trans->labelString[i] != '\0')
	{
		// add "next(" to the buffer
		buffer[j++] = 'n';
		buffer[j++] = 'e';
		buffer[j++] = 'x';
		buffer[j++] = 't';
		buffer[j++] = '(';
	
		// look for the equal sign
		while(trans->labelString[i] != '=')
		{
			// any of these imply we reached the end of the action without finding an '='
			if(trans->labelString[i] == '\0'||trans->labelString[i] == ';'||trans->labelString[i] == '}')
			{
				fprintf(stderr,"\nCould not find '=' in action of transition %s. Check action and rewrite in \"x=y;\" format.\n", trans->labelString);
				return 0;
			}
			
			// any of these would be white space
			else if (trans->labelString[i] == ' '||trans->labelString[i] == '\t'||trans->labelString[i] == '\n')
				i++;

			// a backslash in the string implies a \t or \n that was converted into two characters
			else if (trans->labelString[i] == '\\')
				i+=2;

			// any other letter just copy stright over
			else
				buffer[j++] = trans->labelString[i++];
		}
	
		// add ") := " to the buffer
		buffer[j++] = ')';
		buffer[j++] = ' ';
		buffer[j++] = ':';
		buffer[j++] = '='; i++;
		buffer[j++] = ' ';

		// look for the semicolon
		while(trans->labelString[i] != ';')
		{
			// any of these imply we reached the end of the action without finding a ';'
			if(trans->labelString[i] == '\0' || trans->labelString[i] == ';' || trans->labelString[i] == '}')
			{
				fprintf(stderr,"\nCould not find ';' in action of transition %s. Check action and rewrite in \"x=y;\" format.\n", trans->labelString);
				return 0;
			}

			// any of these would be white space so ignore them
			else if (trans->labelString[i] == ' '||trans->labelString[i] == '\t'||trans->labelString[i] == '\n')
				i++;

			// a backslash in the string implies a \t or \n that was converted into two characters
			else if (trans->labelString[i] == '\\')
				i+=2;

			// otherwise just copy the character directly
			else
				buffer[j++] = trans->labelString[i++];

		}

		// copy the semicolon and add the spacer
		buffer[j++] = trans->labelString[i++]; 
		buffer[j++] = ' ';

	} // end of searching for '\0' loop therefore end of actions


	// we found all the actions so cap off the string, copy the buffer and return 1
	buffer[j] = '\0';
	strcpy(action, buffer);
	return 1;
	

}



/**************************************************************
FIND TRANSITION ACTIONS IN TRANSIION LABEL 
**************************************************************/

int transCondAction(node* trans, char* action)
{
	
	char buffer[500];
	int i = 0; // index for the labelString
	int j = 0; // index for the buffer
	
	// error catching
	if(trans == NULL || trans->labelString == NULL)
		return 0;

	// loop for the whole string until we hit a '{' character indicating a condition action
	while(trans->labelString[i] != '{')
	{
		// if we hit a null character there is no condition actions
		if(trans->labelString[i] == '\0')
			return 0;

		// else if the character wasn't a '/' then go to the next character and check that
		else
			i++;
	}

	// move the index past the '{' that we found and discard it
	i++;


	// if we got here then we have actions so loop for the rest of the actions until we hit a '}'
	while(trans->labelString[i] != '}')
	{

		// add "next(" to the buffer
		buffer[j++] = 'n';
		buffer[j++] = 'e';
		buffer[j++] = 'x';
		buffer[j++] = 't';
		buffer[j++] = '(';
	
		// look for the equal sign
		while(trans->labelString[i] != '=')
		{
			// any of these imply we reached the end of the action without finding an '='
			if(trans->labelString[i] == '\0'||trans->labelString[i] == ';'||trans->labelString[i] == '}')
			{
				fprintf(stderr,"\nCould not find '=' in action of transition %s. Check action and rewrite in \"x=y;\" format.\n", trans->labelString);
				return 0;
			}
			
			// any of these would be white space so ignore them
			else if (trans->labelString[i] == ' '||trans->labelString[i] == '\t'||trans->labelString[i] == '\n')
				i++;

			// a backslash in the string implies a \t or \n that was converted into two characters
			else if (trans->labelString[i] == '\\')
				i+=2;

			// any other letter just copy stright over
			else
				buffer[j++] = trans->labelString[i++];
		}
	
		// add ") := " to the buffer
		buffer[j++] = ')';
		buffer[j++] = ' ';
		buffer[j++] = ':';
		buffer[j++] = '='; i++;
		buffer[j++] = ' ';

		// look for the semicolon
		while(trans->labelString[i] != ';')
		{
			// any of these imply we reached the end of the action without finding a ';'
			if(trans->labelString[i] == '\0' || trans->labelString[i] == ';' || trans->labelString[i] == '}')
			{
				fprintf(stderr,"\nCould not find ';' in action of transition %s. Check action and rewrite in \"x=y;\" format.\n", trans->labelString);
				return 0;
			}

			// any of these would be white space so ignore them
			else if (trans->labelString[i] == ' '||trans->labelString[i] == '\t'||trans->labelString[i] == '\n')
				i++;

			// a backslash in the string implies a \t or \n that was converted into two characters
			else if (trans->labelString[i] == '\\')
				i+=2;

			// otherwise just copy the character directly
			else
				buffer[j++] = trans->labelString[i++];
		}

		// copy the semicolon and add the spacer
		buffer[j++] = trans->labelString[i++]; 
		buffer[j++] = ' ';

	} // end of searching for '\0' loop therefore end of actions


	// we found all the actions so cap off the string, copy the buffer and return 1
	buffer[j] = '\0';
	strcpy(action, buffer);
	return 1;
	


}



/**************************************************************
HELPER FUNCTIONS FOR PRINTING
**************************************************************/

// function to print the appropriate number of tabs for formatting
void align()
{
	int i;

	for(i=0; i<tabs; i++)
		printf("\t");
}

// function to do the insert the proper tabs into a string rather that print them ... not used
void alignStr(char* string)
{
	int i;

	for(i=0; i<tabs; i++)
		strcat(string,"\t");

}

int mkEventCond(char* name, char* event)
{
    char buffer[100];
	int i = 0; // index for the labelString
	int j = 0; // index for the buffer
    int k = 0; // index for labelString
	
    // if we hit a null character there is no condition
    if(event[i] == '\0')
        return 0;
            
    buffer[j++] = '(';
    while (stateList->head->labelString[k] != '\0')
        buffer[j++] = stateList->head->labelString[k++];
    k=0;
    buffer[j++] = 'e';
    buffer[j++] = 'v';
    buffer[j++] = 'e';
    buffer[j++] = 'n';
    buffer[j++] = 't';
    buffer[j++] = 's';
    buffer[j++] = '=';
    
    
	// add the characters to the buffer until we find the closing bracket
	while(event[i] != '\0')
	{
		if ( event[i] == '&' || event[i] == '|' )
		{
            buffer[j++] = ' ';
            buffer[j++] = event[i];
            buffer[j++] = ' ';
            i++;
			// Print event name for next event
            while (stateList->head->labelString[k] != '\0')
                buffer[j++] = stateList->head->labelString[k++];
            
            buffer[j++] = 'e';
            buffer[j++] = 'v';
            buffer[j++] = 'e';
            buffer[j++] = 'n';
            buffer[j++] = 't';
            buffer[j++] = 's';
            buffer[j++] = '=';
            k=0;
		} 
        else if ( event[i] == ' ' )
		{
            i++;
        }    
		else
		{
			buffer[j] = event[i];
			i++;
			j++;
		}
	}
	
	// we found a condition so cap off the string, copy the buffer and return 1
    buffer[j++] = ')';
	buffer[j] = '\0';
	strcpy(event, buffer);
	return 1; 
    
}

// function to add the "Stored_" to the INPUT variable in a condition if we have AND states
int addStored(char* condition)
{
	if (condition == NULL)
		return 0;

	char copy[100];
	char* place;
	char* placeCopy;

	int len	= 0;

	// the copy now contains the same string as the condition(s)
	strcpy(copy, condition);
	
	#ifdef ADDSTOREDDEBUG
	fprintf(stderr, "\n%s -> %s\n", condition, copy);
	#endif

	// loop through the list of data for all the INPUT data
	// when we find an input data, see if its in the string passed in
	// if it is, put the word "Stored_" in front of it
	node* data = dataList->head;
	while (data != NULL)
	{
		// note that strcmp returns 0 if the strings match, hence the negation
		if(!strcmp(data->scope, "INPUT_DATA"))
		{
			#ifdef ADDSTOREDDEBUG
			fprintf(stderr, "Input: %s\n", data->labelString);
			#endif

			// if strstr finds "INPUT" it returns a pointer to it
			place = strstr(condition, data->labelString);
			placeCopy = strstr(copy, data->labelString);
			len = strlen(data->labelString);
			if (	place != NULL &&
			 !(	( *(place-1) >= '0' && *(place-1) <= '9' ) 	||
			 	( *(place-1) >= 'A' && *(place-1) <= 'Z' ) 	||
				( *(place-1) >= 'a' && *(place-1) <= 'z' ) 	||
				( *(place-1) == '_' )			)	&&
			 !(	( *(place+len) >= '0' && *(place+len) <= '9' ) 	||
			 	( *(place+len) >= 'A' && *(place+len) <= 'Z' ) 	||
				( *(place+len) >= 'a' && *(place+len) <= 'z' ) 	||
				( *(place+len) == '_' )			)	)
			{
				#ifdef ADDSTOREDDEBUG
				fprintf(stderr, "%c,%c\n",*(place-1), *(place+len) );
				fprintf(stderr, "Input in cond: %s\n", data->labelString);
				#endif

				*placeCopy = 'S'; placeCopy++;
				*placeCopy = 't'; placeCopy++;
				*placeCopy = 'o'; placeCopy++;
				*placeCopy = 'r'; placeCopy++;
				*placeCopy = 'e'; placeCopy++;
				*placeCopy = 'd'; placeCopy++;
				*placeCopy = '_'; placeCopy++;
				while(*place != '\0')
				{
					*placeCopy = *place;
					placeCopy++;
					place++;
				}
				*placeCopy = '\0';
				#ifdef ADDSTOREDDEBUG
				fprintf(stderr, "%s\n%s\n", condition, copy);
				#endif

			}
		}
	
		data = data->next;
	}

	#ifdef ADDSTOREDDEBUG		
	fprintf(stderr, "Final: %s\n\t%s\n", condition, copy);
	#endif

	strcpy(condition, copy);
				
	return 1;
}



// function to print all of the Stored variables if we have AND states
void storeInputs(node* child)
{
	node* temp = NULL;
	// if we have AND states, print the stored variables again to keep their values
	// but if we're at the last AND state, update the stored variables from the actual INPUTS
	if (ANDstates)
	{
		// find the parent who is the AND_state
		node* ANDparent = child;
		while ((ANDparent != NULL) && (strcmp(ANDparent->type, "AND_STATE"))) 
			ANDparent = ANDparent->parent;
		
		if (ANDparent == NULL) 
			fprintf(stderr, "\n\nWe should have AND states but I couldn't find it when printing the switch statements!\n");

		if(ANDparent->rightSib != NULL)
		{
			if(eventsCnt)
			{
				align();
				printf("next(Stored_%sevents) := Stored_%sevents;\n", stateList->head->labelString, stateList->head->labelString);
			}

			temp = dataList->head;
			while(temp != NULL)
			{
				// note that strcmp returns 0 if the strings match, hence the negation
				if(!strcmp(temp->scope, "INPUT_DATA"))
				{
					align();
					printf("next(Stored_%s) := Stored_%s;\n", temp->labelString, temp->labelString);
				}
				temp = temp->next;
			}
			align();
			printf("next(%sStepDone) := 0;\n", stateList->head->labelString);		
		}
		else
		{
			if(eventsCnt)
			{
				align();
				printf("next(Stored_%sevents) := %sevents;\n", stateList->head->labelString, stateList->head->labelString);
			}
				temp = dataList->head;
			while(temp != NULL)
			{
				// note that strcmp returns 0 if the strings match, hence the negation
				if(!strcmp(temp->scope, "INPUT_DATA"))
				{
					align();
					printf("next(Stored_%s) := %s;\n", temp->labelString, temp->labelString);
				}
				temp = temp->next;
			}
			align();
			printf("next(%sStepDone) := 1;\n", stateList->head->labelString);		
		}
	}
	else
	{
		align();
		printf("next(%sStepDone) := 1;\n", stateList->head->labelString);
	}

}
