/*======================================================================
*** functions.h ***
* (c) Alma L. Juarez-Dominguez, Richard Fanson
* C functions for the creation of nodes, and the addition of attributes 
*  to the respective list. Some functions for string manipulation are
*  also included.
*
* Created: August 2007
* Last modified: August 1, 2011
*======================================================================*/

#include "data.h"
#include "structs.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>



extern list* stateList;
extern list* transList;
extern list* juncList;
extern list* eventList;
extern list* dataList;
extern list* stateTree;

extern int tabs;
extern int tabsMAX;
extern int ANDstates;

extern FILE* tree;
extern FILE* lists;
extern FILE* nodes;



/*****************************************************************************************
***************************     SETUP/TAKEDOWN FUNCTIONS    ******************************
*****************************************************************************************/

/**************************************************************
initialize a new node
**************************************************************/
node * createNode()
{
	node * newNode = (node*)malloc(sizeof(node));

	newNode->id = 0;
	newNode->labelString = NULL;
	newNode->parentId = 0;
	newNode->childId = 0;
	newNode->leftSibId = 0;
	newNode->rightSibId = 0;
	newNode->firstTrans = 0;
	newNode->type = NULL;
	newNode->execOrder = 0;
	newNode->src = 0;
	newNode->dst = 0;
	
	newNode->scope = NULL;
	newNode->triggerData = NULL;

	newNode->next = NULL;
	newNode->parent = NULL;
	newNode->child = NULL;
	newNode->leftSib = NULL;
	newNode->rightSib = NULL;
	
	newNode->min_range = 0;
	newNode->max_range = 0;
	newNode->init_val = 0;

	return newNode;	
}

/**************************************************************
creates a new linked list
**************************************************************/
list * createList(int listType)
{
	list * newList = (list*)malloc(sizeof(list));

	newList->head = NULL;
	newList->type = listType;
	newList->count = 0;

	#ifdef DIAGNOSTICS
	fprintf(stderr, "\nType %d List Created", newList->type);
	#endif

	return newList;
}

/**************************************************************
deletes the contents of a node and the node itself that were allocated
**************************************************************/
int deleteNode(node * toDelete)
{
	if (toDelete == NULL)
		return ERROR;

	free(toDelete->labelString);
	free(toDelete->type);
	free(toDelete->scope);
	free(toDelete->triggerData);
	free(toDelete);

	return SUCCESS;
}

/**************************************************************
deletes an entire list of nodes
**************************************************************/
int deleteList(list * listDel)
{
	node * nodeDel = NULL;
	
	if(listDel == NULL) return ERROR;

	#ifdef DIAGNOSTICS
	fprintf(stderr, "\nDeleting List %d ... %d", listDel->type, listDel->count);
	#endif

	if (listDel->type != STATE_T)
		while(listDel->head != NULL && listDel->type != STATE_T)
		{
			//printNode(listDel->head);	

		
			nodeDel = listDel->head;
			listDel->head = listDel->head->next;
			
			if (deleteNode(nodeDel) == ERROR)
				fprintf(stderr, "\n\nERROR deleting Node %d from List %d\n\n", nodeDel->id, listDel->type);
			
			(listDel->count)--;

			#ifdef DIAGNOSTICS
			fprintf(stderr, " %d", listDel->count);
			#endif
		}
	else
	{
		// do nothing here, nodes in the state tree were already deleted with the state list
		#ifdef DIAGNOSTICS
		fprintf(stderr, " (these nodes were deleted with list 1)");
		#endif
	}

	#ifdef DIAGNOSTICS
	fprintf(stderr, " ... done", listDel->type);
	#endif

	free(listDel);

	return SUCCESS;
}





/*****************************************************************************************
***********************     PRINTING FUNCTIONS FOR DEBUGGING     *************************
*****************************************************************************************/

/**************************************************************
prints all the data in a node
**************************************************************/
int printNode(node * toPrint)
{
	if(toPrint == NULL)
		return ERROR;

	fprintf(nodes, "\nNODE %p:\n  id: %d\n  label: %s\n  parent: %d\n  child: %d\n  leftSib: %d\n  rightSib: %d\n  firstTrans: %d\n  type: %s\n  execOrder: %d\n  src: %d\n  dst: %d\n  scope: %s\n  trigData: %s\n  next: %p\n  parent: %p\n  child: %p\n  leftSib: %p\n  rightSib: %p\n\n", 
	toPrint,
	toPrint->id,
	toPrint->labelString,
	toPrint->parentId,
	toPrint->childId,
	toPrint->leftSibId,
	toPrint->rightSibId,
	toPrint->firstTrans,
	toPrint->type,
	toPrint->execOrder,

	toPrint->src,
	toPrint->dst,

	toPrint->scope,
	toPrint->triggerData,

	toPrint->next,
	toPrint->parent,
	toPrint->child,
	toPrint->leftSib,
	toPrint->rightSib,
	
	toPrint->min_range,
	toPrint->max_range,
	toPrint->init_val);

	return SUCCESS;
}

/**************************************************************
prints all the details of a list
**************************************************************/

int printList(list * toPrint)
{
	node* temp = toPrint->head;

	fprintf(lists, "\n****PRINTING LIST****\n  list: %p\n  head: %p\n  type: %d\n  count: %d\n", toPrint, toPrint->head, toPrint->type, toPrint->count);
	
	while (temp != NULL)
	{
		if (toPrint->type == STATE_L)
			fprintf(lists, "\nSTATE NODE: %p\n  id: %d\n  label: %s\n  parent: %d\n  child: %d\n  leftSib: %d\n  rightSib: %d\n  firstTrans: %d\n  type: %s\n  execOrder: %d\n", temp, temp->id, temp->labelString, temp->parentId, temp->childId, temp->leftSibId, temp->rightSibId, temp->firstTrans, temp->type, temp->execOrder);

		else if (toPrint->type == TRANS_L)
			fprintf(lists, "\nTRANS NODE: %p\n  id: %d\n  label: %s\n  src: %d\n  dst: %d\n  execOrder: %d\n  nextNode: %p\n", temp, temp->id, temp->labelString, temp->src, temp->dst, temp->execOrder, temp->next);

		else if(toPrint->type == JUNC_L)
			fprintf(lists, "\nJUNC NODE: %p\n  id: %d\n  label: %s\n  parent: %d\n  type: %s\n", temp, temp->id, temp->labelString, temp->parentId, temp->type);

		else if(toPrint->type == EVENT_L)
			fprintf(lists, "\nEVENT NODE: %p\n  id: %d\n  name: %s\n  scope: %s\n  trigger: %s\n", temp, temp->id, temp->labelString, temp->scope, temp->triggerData);

		else if(toPrint->type == DATA_L)
			fprintf(lists, "\nDATA NODE: %p\n  id: %d\n  name: %s\n  scope: %s\n  data: %s\n min_range: %d\n max_range: %d\n ", temp, temp->id, temp->labelString, temp->scope, temp->triggerData, temp->min_range, temp->max_range);
		
		temp = temp->next;
	}
	
	fprintf(lists, "*****END OF LIST*****\n\n");
	return SUCCESS;
}





/*****************************************************************************************
********************     LINKED LIST AND POINTER MANIPULATION     ************************
*****************************************************************************************/

/**************************************************************
ADD NODES TO THE LINKED LISTS
**************************************************************/

// adds nodes to the list as they are parsed
int addToList(list * addList, node * addNode)
{
	node * end = findEnd(addList);
	if(end == NULL)
		addList->head = addNode;
	else
		end->next = addNode;

	(addList->count)++;
	return SUCCESS;
}

/**************************************************************
MAKE THE STATE TREE STRUCTURE
**************************************************************/
// makes the heirarchical tree structure for the state nodes AFTER all have been created
int makeTree()
{
	node* addNode = stateList->head; // node that were adding to the tree
	node* relative = NULL; // node used to add the relative links

	// make the head of the tree the first in the state list (should be the "Chart" node)
	stateTree->head = addNode;

	// so far we have a tree with the one "Chart" node with no relatives
	// begin traversing the whole state list to add the nodes

	// set up the links to relatives in every state in the stateList
	while (addNode != NULL)
	{

		// find the parent of the new node and make the link
		if (addNode->parentId != 0)
		{
			relative = stateList->head;

			// find the parent node
			while (relative != NULL && relative->id != addNode->parentId)
				relative = relative->next;

			if (relative == NULL)
			{
				fprintf(stderr, "\n\nerror: parent of %s could not be found when making tree\n\n", addNode->labelString);
				return ERROR;
			}
			
			// make the link
			addNode->parent = relative;
		}

		// find the child of the new node and make the link
		if (addNode->childId != 0)
		{
			relative = stateList->head;

			// find the child node
			while (relative != NULL && relative->id != addNode->childId)
				relative = relative->next;

			if (relative == NULL)
			{
				fprintf(stderr, "\n\nerror: child of %s could not be found when making tree\n\n", addNode->labelString);
				return ERROR;
			}

			// make the link
			addNode->child = relative;
		}

		// find the left sibling of the new node and make the link
		if (addNode->leftSibId != 0)
		{
			relative = stateList->head;

			// find the left sibling node
			while(relative != NULL && relative->id != addNode->leftSibId)
				relative = relative->next;

			if (relative == NULL)
			{
				fprintf(stderr, "\n\nerror: leftSib of %s could not be found when making tree\n\n", addNode->labelString);
				return ERROR;
			}

			// make the link
			addNode->leftSib = relative;
		}

		// find the right sibling of the new node and make the link
		if (addNode->rightSibId != 0)
		{
			relative = stateList->head;

			// find the child node
			while(relative != NULL && relative->id != addNode->rightSibId)
				relative = relative->next;

			if (relative == NULL)
			{
				fprintf(stderr, "\n\nerror: rightSib of %s could not be found when making tree\n\n", addNode->labelString);
				return ERROR;
			}

			// make the link
			addNode->rightSib = relative;
		}
		
		stateTree->count++;
		addNode = addNode->next;
	}


	// print the tree to the console to see the links if this is defined
	#ifdef TREEDEBUG
	node* temp = NULL; // node used for printing and debugging at the end
	fprintf(tree, "\n****PRINTING TREE****\n  list: %p\n  head: %p\n  type: %d\n  count: %d\n", stateTree, stateTree->head, stateTree->type, stateTree->count);

	temp = stateTree->head;
	while(temp != NULL)
	{
		fprintf(tree, "\nSTATE NODE: %p\n  id: %d\n  label: %s\n  parent: %p\n  child: %p\n  leftSib: %p\n  rightSib: %p\n", temp, temp->id, temp->labelString, temp->parent, temp->child, temp->leftSib, temp->rightSib);

		temp = temp->next;
	}

	fprintf(tree, "\n****TREE DONE****\n");
	#endif
	
	return SUCCESS;			
}


/**************************************************************
MAKE THE TRANSITION LINKS TO STATES
**************************************************************/

// makes links for the transition nodes to have pointers to the src (parent) state and dst (child) state
// (and the name of this function pays homage to my summer in BC)
int transLink()
{
	node* trans = transList->head;
	node* state = stateList->head;

	// traverse the entire transition list
	while(trans != NULL)
	{

		// if the transition has a source (ie is not a default transition)
		if(trans->src != 0)
		{
			state = stateList->head;
			
			// find the state matching the source id
			while(state != NULL && trans->src != state->id)
				state = state->next;
			
			// error check to see if we actually hit NULL (only case is if the source is a junction)
			if (state == NULL)
			{
				fprintf(stderr, "\n\nerror: source %d of %p could not be found when making trans link\n\n", trans->src, trans);
				return ERROR;
			}
			
			// make the link
			trans->parent = state;

		}
	
		// if the transition has a destination (every transition should)
		if(trans->dst != 0)
		{
			state = stateList->head;
			
			// find the state matching the destination id
			while(state != NULL && trans->dst != state->id)
				state = state->next;
			
			// error check to see if we actually hit NULL (only case is if the dest is a junction)
			if (state == NULL)
			{
				fprintf(stderr, "\n\nerror: destination %d of %p could not be found when making trans link\n\n", trans->dst, trans);
				return ERROR;
			}
			
			// make the link
			trans->child = state;

		}

		// print for debugging if defined
		#ifdef TRANSDEBUG
		printNode(trans);
		#endif
		
		// repeat for all transitions in the list
		trans = trans->next;
	}

	return SUCCESS;
}

/*****************************************************************************************
***************************     FUNCTIONS FOR YACC PARSING     ***************************
*****************************************************************************************/

/**************************************************************
NODE FIELD SETUP FUNCIONS
**************************************************************/
int addId(node* node, int id)
{
	if(node == NULL || id < 0)
		return ERROR;

	node->id = id;
	return SUCCESS;
}

int addLabelString(list* list, char* label)
{
	if(list == NULL || label == NULL)
		return ERROR;

	removeQuotes(label);	// comment out this line if you want the quotation marks left in

	node* end = findEnd(list);
	end->labelString = strdup(label);		// strdup allocates memory and returns the string (char pointer)

	return SUCCESS;
}

int concatLabelString(list* list, char* addLabel)
{
	if(list == NULL || addLabel == NULL)
		return ERROR;

	// need to free the old string memory and allocate new memory for the bigger string
	// otherwise an error later will occur when trying to free the string memory of a mismatched size
	node* end = findEnd(list);
	char buffer[500];			// make a buffer for the combined string

	strcpy(buffer, end->labelString);	// put the old label in the buffer
	strcat(buffer, addLabel);		// add the new label to the old label in the buffer
	free(end->labelString);			// free the memory from the old label
	addLabelString(list, buffer);		// send the new label pointer to the node with addLabelString

	return SUCCESS;
}

int addRelations(node* node, int parent, int child, int leftSib, int rightSib)
{
	if(node == NULL)
		return ERROR;

	node->parentId = parent;
	node->childId = child;
	node->leftSibId = leftSib;
	node->rightSibId = rightSib;
	return SUCCESS;
}

int addFirstTrans(list* list, int trans)
{
	if(list == NULL || trans < 0)
		return ERROR;

	node* end = findEnd(list);
	end->firstTrans = trans;
	return SUCCESS;
}

int addType(list* list, char* type)
{
	if(list == NULL || type == NULL)
		return ERROR;

	node* end = findEnd(list);
	end->type = strdup(type);
	return SUCCESS;

}

int addExecOrder(list* list, int order)
{
	if(list == NULL || order < 0)
		return ERROR;

	node* end = findEnd(list);
	end->execOrder = order;
	return SUCCESS;
}	

int addSrcId(list* list, int srcId)
{
	if(list == NULL || srcId < 0)
		return ERROR;

	node* end = findEnd(list);
	end->src = srcId;
	return SUCCESS;
}

int addDstId(list* list, int dstId)
{
	if(list == NULL || dstId < 0)
		return ERROR;

	node* end = findEnd(list);
	end->dst = dstId;
	return SUCCESS;
}

int addScope(list* list, char* scope)
{
	if(list == NULL || scope == NULL)
		return ERROR;
	
	node* end = findEnd(list);
	end->scope = strdup(scope);	//the strdup allocates the memory for the string it is freed in deleteNode()
	return SUCCESS;
}

int addTriggerData(list* list, char* triggerData)
{
	if(list == NULL || triggerData == NULL)
		return ERROR;

	removeQuotes(triggerData);	// remove this line to retain quotation marks from input

	node* end = findEnd(list);
	end->triggerData = strdup(triggerData);	//the strdup allocates the memory for the string it is freed in deleteNode()
	return SUCCESS;
}

int addMinMax(list* list, char* min, char* max)
{
	if(list == NULL || min == NULL || max == NULL)
		return ERROR;

	//removeQuotes(min);	// remove this line to retain quotation marks from input
	//removeQuotes(max);	// remove this line to retain quotation marks from input

	node* end = findEnd(list);
	
	end->min_range = FindNum(min);	
	end->max_range = FindNum(max);	
	return SUCCESS;
}

int addInitVal(list* list, char* val_init)
{
	if(list == NULL || val_init == NULL)
		return ERROR;
	
	node* end = findEnd(list);
	
	end->init_val = FindNum(val_init);	
	return SUCCESS;
}



/*****************************************************************************************
********************************     HELPER FUNCTIONS     ********************************
*****************************************************************************************/

int moveStateActions()
{
	node* state = stateList->head;
	int moved = 0;
	while(state != NULL)
	{
		moved += moveActions(state);
		state = state->next;
	}

	fprintf(stderr, "\n%d states had actions\n", moved);
	return 1;
}

/**************************************************************
Parse the State labels to convert to transitions
**************************************************************/

int moveActions(node* state)
{
	char actions[500];
	char* entry = NULL;
	char* exit = NULL;
	char* during = NULL;
	char* newEndLabel = NULL;
	int i = 0;

	// error checking
	if (state == NULL || state->labelString == NULL)
		return 0;

	if ( strstr(state->labelString, "\\n") != NULL )
	{
		fprintf(stderr, "\nState %d has a NL!\n", state->id);
		newEndLabel = strstr(state->labelString, "\\n");

		// see if there is any actions in the string
		if( strstr(state->labelString, "entry:") != NULL )
			entry = strstr(state->labelString, "entry:");
		if( strstr(state->labelString, "en:") != NULL )
			entry = strstr(state->labelString, "en:");
		if( strstr(state->labelString, "exit:") != NULL )
			exit = strstr(state->labelString, "exit:");
		if( strstr(state->labelString, "ex:") != NULL )
			exit = strstr(state->labelString, "ex:");
		if( strstr(state->labelString, "during:") != NULL )
			during = strstr(state->labelString, "during:");
		if( strstr(state->labelString, "du:") != NULL )
			during = strstr(state->labelString, "du:");

		fprintf(stderr, "%s\n", entry);
		fprintf(stderr, "%s\n", exit);
		fprintf(stderr, "%s\n", during);

		if(entry)
			while(*entry != '\0' && entry != exit)
			{
				actions[i] = *entry;
				fprintf(stderr, "%c", *entry);
				i++;
				entry++;
			}
		actions[i] = '\0';
		fprintf(stderr, "\n%s\n", actions);
		


		newEndLabel[0] = '\0';
		return 1;
	}

	else
	{
		return 0;
	}
}


/**************************************************************
Quotation Mark Removing Helper Functions
**************************************************************/

// remove the quotes from a string and copies over the old string with the new one
void removeQuotes(char* string)
{
	char buffer[500];
	if (string == NULL)
		return;

	int i = 0; // for the string
	int j = 0; // for the buffer

	while(string[i] != '\0')
	{
		if(string[i] == '\"')
		{
			i++;
		}
		else
		{
			buffer[j] = string[i];
			i++;
			j++;
		}
	}
	buffer[j] = '\0';
	strcpy(string, buffer);

}
	
// remove the quotes from the label of a node and then frees the old labe and allocates a new one
// may not need this function now that I made one to above function to remove quotes from the string
void removeLabelQuotes(node* node)
{
	char buffer[500];
	int i = 0; // for the string
	int j = 0; // for the buffer
	if (node->labelString != NULL)
	{
		while(node->labelString[i] != '\0')
		{
			if(node->labelString[i] == '\"')
			{
				i++;
			}
			else
			{
				buffer[j] = node->labelString[i];
				i++;
				j++;
			}
		}
		buffer[j] = '\0';
		free(node->labelString);		// free the memory from the old label
		node->labelString = strdup(buffer);	// allocate the new quote-free label
	}
}

/**************************************************************
Quotation Mark Removing and Returning Number that was in Quotes Helper Function
**************************************************************/

// remove the quotes from a string, and obtains the number that was in
// quotes, copying such number over the old string with the new one
int FindNum(char* string)
{
	char buffer[500];
	int pair_quotes=0;
	if (string == NULL)
		return(1);

	int i = 0; // for the string
	int j = 0; // for the buffer

	while(string[i] != '\0')
	{
		if(string[i] == '\"')
		{
			i++;
			pair_quotes++;
		}
		else
		{
			if (pair_quotes==2) 
			{
				buffer[j] = '\0';
				break;
			}	
			else	
			{
				buffer[j] = string[i];
				i++;
				j++;
			}	
		}
	}
	
	return atoi(strcpy(string, buffer));

}

/**************************************************************
Function to find and return the end node of a linked list
**************************************************************/
node * findEnd(list * listGiven)
{
	node * temp = listGiven->head;
	if (temp == NULL)
		return NULL;
	while (temp->next != NULL)
		temp = temp->next;
	return temp;
}

/**************************************************************
Sanity Check function for check accuracy of model
**************************************************************/

// this function contains checks to ensure the model file has been parsed properly
// and data has been properly stored in memory
// prints are sent to the console to help debug the problem
void sanityCheck()
{
	int noGoods = 0;

	// check to make sure there are states present
	if (stateList->count == 0 || stateList->head == NULL)
	{
		fprintf(stderr, "\nNo states appear to have been parsed from model\n\n");
		noGoods++;
	}

	// check to make sure there are transitions present
	if (transList->count == 0 || transList->head == NULL)
	{
		fprintf(stderr, "\nNo transitions appear to have been parsed from model\n\n");
		noGoods++;
	}

	// check to make sure there ARE NOT juncions present ... junctions will be taken care of later
	//if (juncList->count != 0 || juncList->head != NULL)
	//{
	//	fprintf(stderr, "\nJunctions were found in model\nI don't like these yet\n\n");
	//	noGoods++;
	//}

	// check to make sure there ARE
	if ( (eventList->count == 0 || eventList->head == NULL) && (dataList->count == 0 || dataList->head == NULL) )
	{
		fprintf(stderr, "\nNo data or events appear to have been parsed from model\n\n");
		noGoods++;
	}

	if (noGoods)
	{
		fprintf(stderr, "\nMay not have parsed properly. Please check your Stateflow model and try again\n\n\n\n*** Sanity Check ... FAILED ***\n\n");
	}
	
	else
	{
		#ifdef DIAGNOSTICS
		fprintf(stderr, "\n\n*** Sanity Check ... OK ***\n\n");
		#endif
	}


}

int checkAND()
{
	node* state = stateList->head;
	int check = 0;

	while(state != NULL)
	{
		// check to see if any of the states are and states
		if ( state->type != NULL && strstr(state->type, "AND") != NULL )
			check = 1; 

		state = state->next;
	}

	return check;
}


